/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.schema;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import com.datastax.bdp.graph.api.model.*;
import com.datastax.bdp.graph.api.model.EdgeLabel;
import com.datastax.bdp.graph.api.model.Schema;
import com.datastax.bdp.graph.api.model.VertexLabel;
import com.github.misberner.duzzt.annotations.DSLAction;
import com.github.misberner.duzzt.annotations.GenerateEmbeddedDSL;
import org.apache.commons.lang3.StringUtils;
import org.apache.tinkerpop.gremlin.structure.Direction;
import org.javatuples.Pair;

import static com.datastax.bdp.graph.api.schema.SchemaImpl.QM;
import static java.util.stream.Collectors.joining;

/**
 * TODO
 */

@GenerateEmbeddedDSL(
        name = "EdgeLabel",
        syntax = "((single|multiple)? properties? connection* ttl? ifNotExists? create) | (connection+ add) | (properties (add | drop)) | describe | exists | drop"
        )
public class EdgeLabelImpl
{

    private final Schema schema;
    private final String name;

    private EdgeLabel.Builder builder;
    private final List<String> propertyKeys = new ArrayList<>();
    private final List<Pair<String, String>> connections = new ArrayList<>();


    EdgeLabelImpl(Schema schema, String name) {
        this.schema = schema;
        this.name = name;
        builder = schema.buildEdgeLabel(name);
    }

    @DSLAction(displayed = true)
    void connection(String outV, String inV) {
        connections.add(Pair.with(outV, inV));
    }

    @DSLAction(displayed = true)
    void properties(String first, String... properties) {
        propertyKeys.add(first);
        propertyKeys.addAll(Arrays.asList(properties));
    }

    @DSLAction(displayed = true)
    void single() {
        builder = builder.single();
    }

    @DSLAction(displayed = true)
    void multiple() {
        builder = builder.multiple();
    }

    @DSLAction(displayed = true)
    void ifNotExists() {
        builder = builder.ifNotExists();
    }

    @DSLAction(displayed = true)
    void ttl(int timeToLiveInSeconds) {
        builder = builder.ttl(Duration.ofSeconds(timeToLiveInSeconds));
    }

    @DSLAction(displayed = true)
    String describe() {
        String result = describeEdgeLabel();
        String connectStr = describeConnections();
        if (StringUtils.isNotBlank(connectStr)) result+= SchemaImpl.NL + connectStr;
        return result;
    }

    String describeEdgeLabel() {
        EdgeLabel edgeLabel = getEdgeLabel();
        StringBuilder s = new StringBuilder();
        s.append(SchemaImpl.EDGE_LABEL_PREFIX).append(QM).append(edgeLabel.name()).append(QM).append(")");
        if (edgeLabel.cardinality()==Cardinality.Single) {
            s.append(".single()");
        } else {
            s.append(".multiple()");
        }
        if (!edgeLabel.propertyKeys().isEmpty()) {
            s.append(".properties(");
            s.append(edgeLabel.propertyKeys().stream().map(it->QM + it.name() + QM).collect(Collectors.joining(", ")));
            s.append(")");
        }
        if (edgeLabel.ttl().isPresent()) {
            s.append(".ttl(").append(edgeLabel.ttl().get().getSeconds()).append(")");
        }

        s.append(".create()");
        return s.toString();
    }

    String describeConnections() {
        EdgeLabel edgeLabel = getEdgeLabel();
        List<Adjacency> adjacencies = getAdjacencies();
        if (!adjacencies.isEmpty()) {
            StringBuilder s = new StringBuilder();
            s.append(SchemaImpl.EDGE_LABEL_PREFIX).append(QM).append(edgeLabel.name()).append(QM).append(")");
            adjacencies.forEach( adj -> s.append(".connection(").append(QM).append(adj.vertexLabel().name()).append(QM)
                    .append(", ").append(QM).append(adj.otherLabel().name()).append(QM).append(")"));
            s.append(".add()");
            return s.toString();
        } else return "";
    }

    @DSLAction(displayed = true)
    void add()
    {
        EdgeLabel edgeLabel = getEdgeLabel();
        propertyKeys.forEach(it -> edgeLabel.addPropertyKey(it));
        registerConnections();
    }

    @DSLAction(displayed = true)
    void create()
    {
        propertyKeys.forEach(pKey -> builder.addPropertyKeys(pKey));
        builder.add();
        registerConnections();
    }

    @DSLAction(displayed = true)
    boolean exists() {
        return schema.edgeLabel(name) != null;
    }

    private void registerConnections() {
        //We assume that the edge label exists based on the context in which this method is called
        connections.forEach(it -> {
            VertexLabel outV = schema.vertexLabel(it.getValue0());
            if (outV == null) throw new IllegalArgumentException("Vertex label does not exist: " + it.getValue0());
            outV.addAdjacency(name, Direction.OUT, it.getValue1());
        });
    }

    private EdgeLabel getEdgeLabel() {
        EdgeLabel edgeLabel = schema.edgeLabel(name);
        if (edgeLabel == null) throw new IllegalArgumentException("Edge label does not exist: " + name);
        return edgeLabel;
    }

    private List<Adjacency> getAdjacencies() {
        EdgeLabel edgeLabel = getEdgeLabel();
        return schema.vertexLabels().stream().flatMap(
                it -> it.adjacencies().stream().filter(a -> a.direction() == Direction.OUT && a.edgeLabel().equals(edgeLabel))).collect(Collectors.toList());
    }

    @DSLAction(displayed = true)
    void drop()
    {
        if (!propertyKeys.isEmpty())
        {
            for (String propertyKey : propertyKeys)
            {
                getEdgeLabel().dropPropertyKey(propertyKey);
            }
        }
        else
        {
            getEdgeLabel().drop();
        }
    }
}
