/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.schema;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import com.datastax.bdp.graph.api.model.*;
import com.datastax.bdp.graph.api.model.PropertyKey;
import com.datastax.bdp.graph.api.model.Schema;
import com.github.misberner.duzzt.annotations.DSLAction;
import com.github.misberner.duzzt.annotations.GenerateEmbeddedDSL;

import static com.datastax.bdp.graph.api.model.BasicValueType.Point;
import static com.datastax.bdp.graph.api.schema.SchemaImpl.QM;

/**
 * Builder for property keys
 */

@GenerateEmbeddedDSL(
        name = "PropertyKey",
        syntax = "((" +
                "Boolean|" +
                "Int|" +
                "Smallint|" +
                "Bigint|" +
                "Float|" +
                "Double|" +
                "Decimal|" +
                "Varint|" +
                "Timestamp|" +
                "Date|" +
                "Time|" +
                "Duration|" +
                "Text|" +
                "Uuid|" +
                "Inet|" +
                "Blob|" +
                "Point (withGeoBounds|withBounds)|" +
                "Linestring (withGeoBounds|withBounds)|" +
                "Polygon (withGeoBounds|withBounds)" +
                ") (multiple|single)? properties? ttl? ifNotExists? create) | (properties (add|drop)) | describe | exists | drop",
        where = {

        },
        enableAllMethods = false)
public class PropertyKeyImpl
{

    private final Schema schema;
    private final String name;

    private PropertyKey.Builder builder = null;
    private List<String> propertyKeys = new ArrayList<>();

    PropertyKeyImpl(Schema schema, String name) {
        this.schema = schema;
        this.name = name;
    }


    @DSLAction(displayed = true)
    void Boolean() {
        builder = schema.buildPropertyKey(name, BasicValueType.Boolean);
    }

    @DSLAction(displayed = true)
    void Int() {
        builder = schema.buildPropertyKey(name, BasicValueType.Int);
    }

    @DSLAction(displayed = true)
    void Smallint() {
        builder = schema.buildPropertyKey(name, BasicValueType.Smallint);
    }

    @DSLAction(displayed = true)
    void Bigint() {
        builder = schema.buildPropertyKey(name, BasicValueType.Bigint);
    }

    @DSLAction(displayed = true)
    void Float() {
        builder = schema.buildPropertyKey(name, BasicValueType.Float);
    }

    @DSLAction(displayed = true)
    void Double() {
        builder = schema.buildPropertyKey(name, BasicValueType.Double);
    }

    @DSLAction(displayed = true)
    void Decimal() {
        builder = schema.buildPropertyKey(name, BasicValueType.Decimal);
    }

    @DSLAction(displayed = true)
    void Varint() {
        builder = schema.buildPropertyKey(name, BasicValueType.Varint);
    }

    @DSLAction(displayed = true)
    void Timestamp() {
        builder = schema.buildPropertyKey(name, BasicValueType.Timestamp);
    }

    @DSLAction(displayed = true)
    void Date() {
        builder = schema.buildPropertyKey(name, BasicValueType.Date);
    }

    @DSLAction(displayed = true)
    void Time() {
        builder = schema.buildPropertyKey(name, BasicValueType.Time);
    }

    @DSLAction(displayed = true)
    void Duration() {
        builder = schema.buildPropertyKey(name, BasicValueType.Duration);
    }

    @DSLAction(displayed = true)
    void Text() {
        builder = schema.buildPropertyKey(name, BasicValueType.Text);
    }

    @DSLAction(displayed = true)
    void Uuid() {
        builder = schema.buildPropertyKey(name, BasicValueType.Uuid);
    }

    @DSLAction(displayed = true)
    void Inet() {
        builder = schema.buildPropertyKey(name, BasicValueType.Inet);
    }

    @DSLAction(displayed = true)
    void Blob() {
        builder = schema.buildPropertyKey(name, BasicValueType.Blob);
    }

    @DSLAction(displayed = true)
    void Point() {
        builder = schema.buildPropertyKey(name, Point);
    }

    @DSLAction(displayed = true)
    void Linestring() {
        builder = schema.buildPropertyKey(name, BasicValueType.Linestring);
    }

    @DSLAction(displayed = true)
    void Polygon() {
        builder = schema.buildPropertyKey(name, BasicValueType.Polygon);
    }

    @DSLAction(displayed = true)
    void multiple() {
        builder = builder.multiple();
    }

    @DSLAction(displayed = true)
    void single() {
        builder = builder.single();
    }

    @DSLAction(displayed = true)
    void properties(String first, String... properties) {
        propertyKeys.add(first);
        if (properties != null) propertyKeys.addAll(Arrays.asList(properties));
    }

    @DSLAction(displayed = true)
    void ifNotExists() {
        builder = builder.ifNotExists();
    }

    @DSLAction(displayed = true)
    void ttl(int timeToLiveInSeconds) {
        builder = builder.ttl(Duration.ofSeconds(timeToLiveInSeconds));
    }

    @DSLAction(displayed = true)
    void create() {
        propertyKeys.forEach(it->builder.addPropertyKeys(it));
        builder.add();
    }

    @DSLAction(displayed = true)
    void add() {
        propertyKeys.forEach(it->getPropertyKey().addPropertyKey(it));
    }

    @DSLAction(displayed = true)
    void drop()
    {
        if (!propertyKeys.isEmpty())
        {
            propertyKeys.forEach(it -> getPropertyKey().dropPropertyKey(it));
        }
        else
        {
            getPropertyKey().drop();
        }
    }

    @DSLAction(displayed = true)
    String describe() {
        return describe(true);
    }

    private String describe(boolean metaProperties) {
        PropertyKey propertyKey = getPropertyKey();
        StringBuilder s = new StringBuilder();
        s.append(SchemaImpl.PROPERTY_KEY_PREFIX).append(QM).append(propertyKey.name()).append(QM).append(")");
        s.append("." + propertyKey.dataType().toString() + "()");

        List<PropertyKey.Validator<?>> validators = propertyKey.validators();
        for(PropertyKey.Validator validator : validators) {
            s.append(validator.describe());
        }

        if (propertyKey.cardinality()==Cardinality.Single) {
            s.append(".single()");
        } else {
            s.append(".multiple()");
        }
        if (!propertyKey.propertyKeys().isEmpty() && metaProperties) {
            s.append(".properties(");
            s.append(propertyKey.propertyKeys().stream().map(it->QM + it.name() + QM).collect(Collectors.joining(", ")));
            s.append(")");
        }
        if (propertyKey.ttl().isPresent()) {
            s.append(".ttl(").append(propertyKey.ttl().get().getSeconds()).append(")");
        }

        s.append(".create()");
        return s.toString();
    }


    public String describeCoreDefinition() {
        return describe(false);
    }

    public String describeMetaProperties() {
        PropertyKey propertyKey = getPropertyKey();
        if (!propertyKey.propertyKeys().isEmpty()) {
            StringBuilder s = new StringBuilder();
            s.append(SchemaImpl.PROPERTY_KEY_PREFIX).append(QM).append(propertyKey.name()).append(QM).append(")");

            if (!propertyKey.propertyKeys().isEmpty()) {
                s.append(".properties(");
                s.append(propertyKey.propertyKeys().stream().map(it -> QM + it.name() + QM).collect(Collectors.joining(", ")));
                s.append(")");
            }

            s.append(".add()");
            return s.toString();
        }
        return null;
    }


    @DSLAction(displayed = true)
    boolean exists() {
        return schema.propertyKey(name) != null;
    }

    private PropertyKey getPropertyKey() {
        PropertyKey propertyKey = schema.propertyKey(name);
        if (propertyKey == null) throw new IllegalArgumentException("Property key does not exist: " + name);
        return propertyKey;
    }

    @DSLAction(displayed = true)
    void withGeoBounds() {
        builder.geoBounds();
    }

    @DSLAction(displayed = true)
    void withBounds(double minX, double minY, double maxX, double maxY) {
        builder.bounds(minX, minY, maxX, maxY);
    }

}
