/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.datastax.bdp.graph.api.Identified;
import com.datastax.bdp.graph.api.Named;
import com.datastax.bdp.graph.api.id.SchemaId;

import javax.xml.validation.Validator;

/**
 * Represents the definition of property key that can be used on vertex or edge labels.
 */
public interface PropertyKey extends Named, Identified {


    /**
     * Unique identifier for this entity.
     *
     * @return Unique id for this entity
     */
    SchemaId id();

    /**
     * @return The data type of the property's value.
     */
    ValueType dataType();

    /**
     * @return The time to live for the relation type. Once expired the relation will be automatically removed.
     */
    Optional<Duration> ttl();

    /**
     * @return The cardinality of this property key, {@link Cardinality#Single} means only one property of this key is allowed
     * on a vertex. {@link Cardinality#Multiple} allows multiple on a vertex. Edges only support single.
     */
    Cardinality cardinality();

    /**
     * @return the property keys associated with this property key.
     */
    Set<? extends PropertyKey> propertyKeys();

    /**
     * Register that a particular property key may be used on properties with this property key.
     * @param name The property key name
     */
    PropertyKey addPropertyKey(String name);

    /**
     * Drop registration that a particular property key may be used on properties with this particular key.
     * @param name The property key name
     */
    void dropPropertyKey(String name);

    /**
     * Drop the property key from the schema
     */
    void drop();

    /**
     * @return Returns the list of validators associated with this property key.
     */
    List<PropertyKey.Validator<?>> validators();

    /**
     * A validator is a user configurable object that restricts the valid values of a property key. The concrete implementation
     * depends on the type of property.
     */
    interface Validator<T> {



        enum Type {
            Bounds,
            GeoBounds,
        }

        /**
         * Validate the argument. If it is invalid then an exception is thrown.
         * @param o The argument to validate.
         * @throws IllegalArgumentException
         */
        void validate(T o) throws IllegalArgumentException;

        /**
         * @return The type of validator
         */
        Type type();

        /**
         * @return The validator description
         */
        String describe();
    }


    /**
     * Builder interface for properties.
     */
    interface Builder {

        /**
         * Sets the cardinality for this property key. If the cardinality is set to {@link Cardinality#Single} then only
         * a single value may be associated with this key. If the cardinality
         * is {@link Cardinality#Multiple} then multiple values can be associated.
         *
         * The default cardinality is {@link Cardinality#Multiple}.
         *
         * @param cardinality The cardinality of this property key.
         * @return This builder.
         */
        Builder cardinality(Cardinality cardinality);

        /**
         * Sets the cardinality for this property key to single.
         *
         * @return This builder
         * @see #cardinality(Cardinality)
         */
        default Builder single() {
            return cardinality(Cardinality.Single);
        }

        /**
         * Sets the cardinality for this property key to multiple (which is the default).
         *
         * @return This builder
         * @see #cardinality(Cardinality)
         */
        default Builder multiple() {
            return cardinality(Cardinality.Multiple);
        }

        /**
         * @param ttl The time to live for the property key. Once expired the edge will be automatically removed.
         * @return the builder.
         */
        Builder ttl(Duration ttl);

        /**
         * @return if the property key already exists then just return it.
         */
        Builder ifNotExists();

        /**
         * Create and add the property type to the schema.
         * @return the added property type.
         */
        PropertyKey add();

        /**
         * Add property keys to the vertex label
         *
         * @param propertyKeys The property key.
         */
        Builder addPropertyKeys(String... propertyKeys);

        /**
         * Applies to Point type only. Sets the parameters for range checking data.
         * @param minX The minimum X value
         * @param minY The minimum Y value
         * @param maxX The max X value
         * @param maxY The max Y value
         */
        Builder bounds(double minX, double minY, double maxX, double maxY);

        /**
         * Applies to Point type only. Restricts values to geoBounds range.
         */
        Builder geoBounds();
    }


}
