/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model;

import java.util.Set;

import org.apache.tinkerpop.gremlin.structure.Direction;

import com.datastax.bdp.graph.api.Identified;
import com.datastax.bdp.graph.api.Named;

/**
 * Edge indexes allow quick retrieval of edges with certain properties associated with a particular vertex by gremlin queries.
 */
public interface EdgeIndex extends Named, Identified
{

    /**
     * @return The {@link PropertyKey}s by which the edges of this index are indexed
     */
    Set<? extends PropertyKey> propertyKeys();

    /**
     *
     * @return The {@link VertexLabel} that this index is on.
     */
    VertexLabel vertexLabel();

    /**
     * @return the {@link EdgeLabel} of edges that will be indexed.
     */
    EdgeLabel edgeLabel();

    /**
     * @return the direction in which edges are indexed.
     */
    Direction direction();

    /**
     * Drops this index.
     */
    void drop();


    interface Builder {

        /**
         * @return Create and add this edge index to the schema.
         */
        EdgeIndex add();

        /**
         * Set the direction in which this index should be apply. Defaults to Both.
         * @param direction The direction for this index
         * @return This builder
         */
        Builder direction(Direction direction);

        /**
         * Sets the direction of the index in the outgoing direction
         * @return This builder
         * @see #direction(Direction)
         */
        default Builder out() {
            return direction(Direction.OUT);
        }

        /**
         * Sets the direction of the index in the incoming direction
         * @return This builder
         * @see #direction(Direction)
         */
        default Builder in() {
            return direction(Direction.IN);
        }

        /**
         * Sets the direction of the index in both directions
         * @return This builder
         * @see #direction(Direction)
         */
        default Builder both() {
            return direction(Direction.BOTH);
        }

        /**
         * Indexes edges by the given property key
         *
         * @param propertyKey The property key to be indexed on.
         * @return This builder.
         */
        Builder byPropertyKey(String propertyKey);

        /**
         * @return if the edge index already exists then just return it.
         */
        Builder ifNotExists();
    }

}
