/*
 * Decompiled with CFR 0.152.
 */
package io.nosqlbench.readers;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;

public final class ReaderUtils {
    private static final int MAX_REASONABLE_DIMENSION = 0x100000;

    private ReaderUtils() {
    }

    static int computeVectorCount(Path filePath, long fileSize, int recordSize, int elementWidth) throws IOException {
        if (recordSize <= 0) {
            throw new IOException("Record size must be positive for file: " + String.valueOf(filePath));
        }
        long recordSizeLong = recordSize;
        if (fileSize % recordSizeLong != 0L) {
            String errorMessage;
            block10: {
                if (elementWidth > 0) {
                    try {
                        EndianCheckResult endianCheck = ReaderUtils.checkXvecEndianness(filePath, elementWidth);
                        if (endianCheck.isEndianMismatch()) {
                            errorMessage = "File " + String.valueOf(filePath) + " appears to use big-endian byte order, but little-endian is expected. The file size " + fileSize + " is not a multiple of the expected record size " + recordSizeLong + " when interpreted as little-endian. However, the file appears to be readable when interpreted as big-endian (dimension=" + endianCheck.getBigEndianDimension() + ", vectors=" + endianCheck.getBigEndianVectorCount() + "). Consider converting the file to little-endian format.";
                            break block10;
                        }
                        if (!endianCheck.isLittleEndianValid() && !endianCheck.isBigEndianValid()) {
                            errorMessage = "File size " + fileSize + " is not a multiple of record size " + recordSizeLong + " for file: " + String.valueOf(filePath);
                            String reason = endianCheck.getLittleEndianFailureReason();
                            if (reason != null) {
                                errorMessage = (String)errorMessage + ". Additionally, dimension header validation failed: " + reason;
                            }
                            break block10;
                        }
                        errorMessage = "File size " + fileSize + " is not a multiple of record size " + recordSizeLong + " for file: " + String.valueOf(filePath);
                    }
                    catch (IOException e) {
                        errorMessage = "File size " + fileSize + " is not a multiple of record size " + recordSizeLong + " for file: " + String.valueOf(filePath);
                    }
                } else {
                    errorMessage = "File size " + fileSize + " is not a multiple of record size " + recordSizeLong + " for file: " + String.valueOf(filePath);
                }
            }
            throw new IOException(errorMessage);
        }
        long vectorCount = fileSize / recordSizeLong;
        if (vectorCount > Integer.MAX_VALUE) {
            throw new IOException("Vector count " + vectorCount + " exceeds maximum supported size 2147483647 for file: " + String.valueOf(filePath));
        }
        return (int)vectorCount;
    }

    public static EndianCheckResult checkXvecEndianness(Path filePath, int elementWidth) throws IOException {
        if (elementWidth <= 0) {
            throw new IOException("Element width must be positive for file: " + String.valueOf(filePath));
        }
        EndianValidation little = ReaderUtils.evaluateXvecLayout(filePath, elementWidth, ByteOrder.LITTLE_ENDIAN);
        EndianValidation big = ReaderUtils.evaluateXvecLayout(filePath, elementWidth, ByteOrder.BIG_ENDIAN);
        if (!little.valid && !big.valid) {
            String reason = little.failureReason != null ? little.failureReason : big.failureReason;
            throw new IOException("Unable to interpret dimension header for file " + String.valueOf(filePath) + (String)(reason != null ? ": " + reason : ""));
        }
        return new EndianCheckResult(little, big);
    }

    private static EndianValidation evaluateXvecLayout(Path filePath, int elementWidth, ByteOrder order) throws IOException {
        try (FileChannel channel = FileChannel.open(filePath, StandardOpenOption.READ);){
            long recordSize;
            long fileSize = channel.size();
            if (fileSize < 4L) {
                EndianValidation endianValidation = EndianValidation.invalid("File too small to contain dimension header");
                return endianValidation;
            }
            ByteBuffer dimBuffer = ByteBuffer.allocate(4);
            int bytesRead = channel.read(dimBuffer, 0L);
            if (bytesRead != 4) {
                EndianValidation endianValidation = EndianValidation.invalid("Failed to read dimension header");
                return endianValidation;
            }
            dimBuffer.flip();
            dimBuffer.order(order);
            int dimension = dimBuffer.getInt();
            if (dimension <= 0 || dimension > 0x100000) {
                EndianValidation endianValidation = EndianValidation.invalid("Dimension " + dimension + " out of range");
                return endianValidation;
            }
            try {
                long vectorBytes = Math.multiplyExact((long)dimension, (long)elementWidth);
                recordSize = Math.addExact(4L, vectorBytes);
            }
            catch (ArithmeticException e) {
                EndianValidation endianValidation = EndianValidation.invalid("Vector record size overflow");
                if (channel != null) {
                    channel.close();
                }
                return endianValidation;
            }
            if (recordSize <= 4L) {
                EndianValidation e = EndianValidation.invalid("Record size " + recordSize + " too small");
                return e;
            }
            if (recordSize > fileSize) {
                EndianValidation e = EndianValidation.invalid("Record size " + recordSize + " exceeds file length");
                return e;
            }
            if (fileSize % recordSize != 0L) {
                EndianValidation e = EndianValidation.invalid("File size " + fileSize + " is not a multiple of record size " + recordSize);
                return e;
            }
            long vectorCount = fileSize / recordSize;
            if (vectorCount <= 0L) {
                EndianValidation endianValidation = EndianValidation.invalid("No vectors detected");
                return endianValidation;
            }
            if (vectorCount > 1L) {
                long lastOffset;
                dimBuffer.clear();
                try {
                    lastOffset = Math.multiplyExact(vectorCount - 1L, recordSize);
                }
                catch (ArithmeticException e) {
                    EndianValidation endianValidation = EndianValidation.invalid("Vector offset overflow");
                    if (channel != null) {
                        channel.close();
                    }
                    return endianValidation;
                }
                int lastBytes = channel.read(dimBuffer, lastOffset);
                if (lastBytes != 4) {
                    EndianValidation endianValidation = EndianValidation.invalid("Failed to read trailing dimension");
                    return endianValidation;
                }
                dimBuffer.flip();
                dimBuffer.order(order);
                int trailingDimension = dimBuffer.getInt();
                if (trailingDimension != dimension) {
                    EndianValidation endianValidation = EndianValidation.invalid("Inconsistent dimension detected at final record: " + trailingDimension);
                    return endianValidation;
                }
            }
            EndianValidation endianValidation = EndianValidation.valid(dimension, vectorCount);
            return endianValidation;
        }
    }

    public static final class EndianCheckResult {
        private final EndianValidation little;
        private final EndianValidation big;

        private EndianCheckResult(EndianValidation little, EndianValidation big) {
            this.little = little;
            this.big = big;
        }

        public boolean isLittleEndianValid() {
            return this.little.valid;
        }

        public boolean isBigEndianValid() {
            return this.big.valid;
        }

        public int getLittleEndianDimension() {
            return this.little.dimension;
        }

        public long getLittleEndianVectorCount() {
            return this.little.vectorCount;
        }

        public int getBigEndianDimension() {
            return this.big.dimension;
        }

        public long getBigEndianVectorCount() {
            return this.big.vectorCount;
        }

        public long getVectorCount() {
            return this.little.vectorCount;
        }

        public String getLittleEndianFailureReason() {
            return this.little.failureReason;
        }

        public String getBigEndianFailureReason() {
            return this.big.failureReason;
        }

        public boolean isEndianMismatch() {
            return !this.little.valid && this.big.valid;
        }
    }

    private static final class EndianValidation {
        private final boolean valid;
        private final int dimension;
        private final long vectorCount;
        private final String failureReason;

        private EndianValidation(boolean valid, int dimension, long vectorCount, String failureReason) {
            this.valid = valid;
            this.dimension = dimension;
            this.vectorCount = vectorCount;
            this.failureReason = failureReason;
        }

        private static EndianValidation valid(int dimension, long vectorCount) {
            return new EndianValidation(true, dimension, vectorCount, null);
        }

        private static EndianValidation invalid(String reason) {
            return new EndianValidation(false, 0, 0L, reason);
        }
    }
}

