package io.quarkiverse.mcp.server;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

/**
 * Response to a {@code tools/call} request from the client.
 *
 * @param isError {@code true} if the tool call ended in an error
 * @param content the list of content items (must not be {@code null})
 * @param structuredContent the optional structured result of the tool call
 * @param _meta the optional metadata
 */
public record ToolResponse(boolean isError,
        List<? extends Content> content,
        Object structuredContent,
        Map<MetaKey, Object> _meta) {

    /**
     * @param <C>
     * @param content
     * @return a successful response with the specified content items
     */
    @SafeVarargs
    public static <C extends Content> ToolResponse success(C... content) {
        return new ToolResponse(false, Arrays.asList(content));
    }

    /**
     * @param <C>
     * @param content
     * @return a successful response with the specified content items
     */
    public static <C extends Content> ToolResponse success(List<C> content) {
        return new ToolResponse(false, content);
    }

    /**
     * @param message
     * @return an unsuccessful response with single text content item
     */
    public static ToolResponse error(String message) {
        return new ToolResponse(true, List.of(new TextContent(message)));
    }

    /**
     * @param message
     * @return a successful response with single text content item
     */
    public static ToolResponse success(String message) {
        return new ToolResponse(false, List.of(new TextContent(message)));
    }

    /**
     * @param structuredContent
     * @return an unsuccessful response with structured content
     */
    public static ToolResponse structuredError(Object structuredContent) {
        return new ToolResponse(true, null, structuredContent, null);
    }

    /**
     * @param structuredContent
     * @return a successful response with structured content
     */
    public static ToolResponse structuredSuccess(Object structuredContent) {
        return new ToolResponse(false, null, structuredContent, null);
    }

    public ToolResponse(boolean isError, List<? extends Content> content, Map<MetaKey, Object> _meta) {
        this(isError, content, null, _meta);
    }

    public ToolResponse(boolean isError, List<? extends Content> content) {
        this(isError, content, null);
    }

    public ToolResponse {
        if (content == null && structuredContent == null) {
            throw new IllegalArgumentException("content must not be null");
        }
    }

    public Content firstContent() {
        if (content == null || content.isEmpty()) {
            throw new NoSuchElementException();
        }
        return content.get(0);
    }

}
