/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs;

import com.obs.services.exception.ObsException;
import com.obs.services.model.AbortMultipartUploadRequest;
import com.obs.services.model.CompleteMultipartUploadRequest;
import com.obs.services.model.CopyObjectRequest;
import com.obs.services.model.CopyObjectResult;
import com.obs.services.model.CopyPartRequest;
import com.obs.services.model.CopyPartResult;
import com.obs.services.model.DeleteObjectsRequest;
import com.obs.services.model.GetObjectMetadataRequest;
import com.obs.services.model.InitiateMultipartUploadRequest;
import com.obs.services.model.InitiateMultipartUploadResult;
import com.obs.services.model.KeyAndVersion;
import com.obs.services.model.ListObjectsRequest;
import com.obs.services.model.ObjectListing;
import com.obs.services.model.ObjectMetadata;
import com.obs.services.model.ObsObject;
import com.obs.services.model.PartEtag;
import com.obs.services.model.PutObjectRequest;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.fs.obs.OBSCommonUtils;
import org.apache.hadoop.fs.obs.OBSFileStatus;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.fs.obs.RenameFailedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class OBSObjectBucketUtils {
    private static final Logger LOG = LoggerFactory.getLogger(OBSObjectBucketUtils.class);

    private OBSObjectBucketUtils() {
    }

    static boolean renameBasedOnObject(OBSFileSystem owner, Path src, Path dst) throws RenameFailedException, FileNotFoundException, IOException, ObsException {
        String srcKey = OBSCommonUtils.pathToKey(owner, src);
        String dstKey = OBSCommonUtils.pathToKey(owner, dst);
        if (srcKey.isEmpty()) {
            LOG.error("rename: src [{}] is root directory", (Object)src);
            throw new IOException(src + " is root directory");
        }
        FileStatus srcStatus = owner.getFileStatus(src);
        try {
            FileStatus dstStatus = owner.getFileStatus(dst);
            if (dstStatus.isDirectory()) {
                String newDstKey = OBSCommonUtils.maybeAddTrailingSlash(dstKey);
                String filename = srcKey.substring(OBSCommonUtils.pathToKey(owner, src.getParent()).length() + 1);
                dstKey = newDstKey = newDstKey + filename;
                dstStatus = owner.getFileStatus(OBSCommonUtils.keyToPath(dstKey));
                if (dstStatus.isDirectory()) {
                    throw new RenameFailedException(src, dst, "new destination is an existed directory").withExitCode(false);
                }
                throw new RenameFailedException(src, dst, "new destination is an existed file").withExitCode(false);
            }
            if (srcKey.equals(dstKey)) {
                LOG.warn("rename: src and dest refer to the same file or directory: {}", (Object)dst);
                return true;
            }
            throw new RenameFailedException(src, dst, "destination is an existed file").withExitCode(false);
        }
        catch (FileNotFoundException e) {
            LOG.debug("rename: destination path {} not found", (Object)dst);
            OBSObjectBucketUtils.checkDestinationParent(owner, src, dst);
            if (dstKey.startsWith(srcKey) && dstKey.charAt(srcKey.length()) == '/') {
                LOG.error("rename: dest [{}] cannot be a descendant of src [{}]", (Object)dst, (Object)src);
                return false;
            }
            if (srcStatus.isFile()) {
                LOG.debug("rename: renaming file {} to {}", (Object)src, (Object)dst);
                OBSObjectBucketUtils.renameFile(owner, srcKey, dstKey, srcStatus);
            } else {
                LOG.debug("rename: renaming directory {} to {}", (Object)src, (Object)dst);
                dstKey = OBSCommonUtils.maybeAddTrailingSlash(dstKey);
                srcKey = OBSCommonUtils.maybeAddTrailingSlash(srcKey);
                OBSObjectBucketUtils.renameFolder(owner, srcKey, dstKey);
            }
            if (src.getParent() != dst.getParent()) {
                OBSObjectBucketUtils.createFakeDirectoryIfNecessary(owner, src.getParent());
            }
            return true;
        }
    }

    private static void checkDestinationParent(OBSFileSystem owner, Path src, Path dst) throws IOException {
        Path parent = dst.getParent();
        if (!OBSCommonUtils.pathToKey(owner, parent).isEmpty()) {
            try {
                FileStatus dstParentStatus = owner.getFileStatus(dst.getParent());
                if (!dstParentStatus.isDirectory()) {
                    throw new ParentNotDirectoryException("destination parent [" + dst.getParent() + "] is not a directory");
                }
            }
            catch (FileNotFoundException e2) {
                throw new RenameFailedException(src, dst, "destination has no parent ");
            }
        }
    }

    private static void renameFile(OBSFileSystem owner, String srcKey, String dstKey, FileStatus srcStatus) throws IOException {
        long startTime = System.nanoTime();
        OBSObjectBucketUtils.copyFile(owner, srcKey, dstKey, srcStatus.getLen());
        OBSObjectBucketUtils.objectDelete(owner, srcStatus, false);
        if (LOG.isDebugEnabled()) {
            long delay = System.nanoTime() - startTime;
            LOG.debug("OBSFileSystem rename: , {src=" + srcKey + ", dst=" + dstKey + ", delay=" + delay + "}");
        }
    }

    static boolean objectDelete(OBSFileSystem owner, FileStatus status, boolean recursive) throws IOException {
        Path f = status.getPath();
        String key = OBSCommonUtils.pathToKey(owner, f);
        if (status.isDirectory()) {
            LOG.debug("delete: Path is a directory: {} - recursive {}", (Object)f, (Object)recursive);
            key = OBSCommonUtils.maybeAddTrailingSlash(key);
            if (!key.endsWith("/")) {
                key = key + "/";
            }
            boolean isEmptyDir = OBSCommonUtils.isFolderEmpty(owner, key);
            if (key.equals("/")) {
                return OBSCommonUtils.rejectRootDirectoryDelete(owner.getBucket(), isEmptyDir, recursive);
            }
            if (!recursive && !isEmptyDir) {
                throw new PathIsNotEmptyDirectoryException(f.toString());
            }
            if (isEmptyDir) {
                LOG.debug("delete: Deleting fake empty directory {} - recursive {}", (Object)f, (Object)recursive);
                OBSCommonUtils.deleteObject(owner, key);
            } else {
                LOG.debug("delete: Deleting objects for directory prefix {} - recursive {}", (Object)f, (Object)recursive);
                OBSObjectBucketUtils.deleteNonEmptyDir(owner, recursive, key);
            }
        } else {
            LOG.debug("delete: Path is a file");
            OBSCommonUtils.deleteObject(owner, key);
        }
        Path parent = f.getParent();
        if (parent != null) {
            OBSObjectBucketUtils.createFakeDirectoryIfNecessary(owner, parent);
        }
        return true;
    }

    static void renameFolder(OBSFileSystem owner, String srcKey, String dstKey) throws IOException {
        long startTime = System.nanoTime();
        ArrayList<KeyAndVersion> keysToDelete = new ArrayList<KeyAndVersion>();
        OBSObjectBucketUtils.createFakeDirectory(owner, dstKey);
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(owner.getBucket());
        request.setPrefix(srcKey);
        request.setMaxKeys(owner.getMaxKeys());
        ObjectListing objects = OBSCommonUtils.listObjects(owner, request);
        LinkedList<Future<CopyObjectResult>> copyfutures = new LinkedList<Future<CopyObjectResult>>();
        while (true) {
            for (ObsObject summary : objects.getObjects()) {
                if (summary.getObjectKey().equals(srcKey)) continue;
                keysToDelete.add(new KeyAndVersion(summary.getObjectKey()));
                String newDstKey = dstKey + summary.getObjectKey().substring(srcKey.length());
                copyfutures.add(OBSObjectBucketUtils.copyFileAsync(owner, summary.getObjectKey(), newDstKey, summary.getMetadata().getContentLength()));
                if (keysToDelete.size() != owner.getMaxEntriesToDelete()) continue;
                OBSObjectBucketUtils.waitAllCopyFinished(copyfutures);
                copyfutures.clear();
            }
            if (!objects.isTruncated()) {
                if (keysToDelete.isEmpty()) break;
                OBSObjectBucketUtils.waitAllCopyFinished(copyfutures);
                copyfutures.clear();
                break;
            }
            objects = OBSCommonUtils.continueListObjects(owner, objects);
        }
        keysToDelete.add(new KeyAndVersion(srcKey));
        DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(owner.getBucket());
        deleteObjectsRequest.setKeyAndVersions(keysToDelete.toArray(new KeyAndVersion[0]));
        OBSCommonUtils.deleteObjects(owner, deleteObjectsRequest);
        if (LOG.isDebugEnabled()) {
            long delay = System.nanoTime() - startTime;
            LOG.debug("OBSFileSystem rename: , {src=" + srcKey + ", dst=" + dstKey + ", delay=" + delay + "}");
        }
    }

    private static void waitAllCopyFinished(List<Future<CopyObjectResult>> copyFutures) throws IOException {
        try {
            for (Future<CopyObjectResult> copyFuture : copyFutures) {
                copyFuture.get();
            }
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while copying objects (copy)");
            throw new InterruptedIOException("Interrupted while copying objects (copy)");
        }
        catch (ExecutionException e) {
            for (Future<CopyObjectResult> future : copyFutures) {
                future.cancel(true);
            }
            throw OBSCommonUtils.extractException("waitAllCopyFinished", copyFutures.toString(), e);
        }
    }

    protected static ObjectMetadata getObjectMetadata(OBSFileSystem owner, String key) {
        GetObjectMetadataRequest request = new GetObjectMetadataRequest();
        request.setBucketName(owner.getBucket());
        request.setObjectKey(key);
        if (owner.getSse().isSseCEnable()) {
            request.setSseCHeader(owner.getSse().getSseCHeader());
        }
        ObjectMetadata meta = owner.getObsClient().getObjectMetadata(request);
        owner.getSchemeStatistics().incrementReadOps(1);
        return meta;
    }

    static ObjectMetadata newObjectMetadata(long length) {
        ObjectMetadata om = new ObjectMetadata();
        if (length >= 0L) {
            om.setContentLength(Long.valueOf(length));
        }
        return om;
    }

    private static void deleteNonEmptyDir(OBSFileSystem owner, boolean recursive, String key) throws IOException {
        String delimiter = recursive ? null : "/";
        ListObjectsRequest request = OBSCommonUtils.createListObjectsRequest(owner, key, delimiter);
        ObjectListing objects = OBSCommonUtils.listObjects(owner, request);
        ArrayList<KeyAndVersion> keys = new ArrayList<KeyAndVersion>(objects.getObjects().size());
        while (true) {
            for (ObsObject summary : objects.getObjects()) {
                if (summary.getObjectKey().equals(key)) continue;
                keys.add(new KeyAndVersion(summary.getObjectKey()));
                LOG.debug("Got object to delete {}", (Object)summary.getObjectKey());
                if (keys.size() != owner.getMaxEntriesToDelete()) continue;
                OBSCommonUtils.removeKeys(owner, keys, true, true);
            }
            if (!objects.isTruncated()) break;
            objects = OBSCommonUtils.continueListObjects(owner, objects);
        }
        keys.add(new KeyAndVersion(key));
        OBSCommonUtils.removeKeys(owner, keys, false, true);
    }

    static void createFakeDirectoryIfNecessary(OBSFileSystem owner, Path f) throws IOException, ObsException {
        String key = OBSCommonUtils.pathToKey(owner, f);
        if (!key.isEmpty() && !owner.exists(f)) {
            LOG.debug("Creating new fake directory at {}", (Object)f);
            OBSObjectBucketUtils.createFakeDirectory(owner, key);
        }
    }

    static void createFakeDirectory(OBSFileSystem owner, String objectName) throws ObsException, IOException {
        String newObjectName = objectName;
        newObjectName = OBSCommonUtils.maybeAddTrailingSlash(newObjectName);
        OBSObjectBucketUtils.createEmptyObject(owner, newObjectName);
    }

    private static void createEmptyObject(OBSFileSystem owner, String objectName) throws ObsException, IOException {
        for (int retryTime = 1; retryTime < 3; ++retryTime) {
            try {
                OBSObjectBucketUtils.innerCreateEmptyObject(owner, objectName);
                return;
            }
            catch (ObsException e) {
                LOG.warn("Failed to create empty object [{}], retry time [{}], exception [{}]", new Object[]{objectName, retryTime, e});
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        OBSObjectBucketUtils.innerCreateEmptyObject(owner, objectName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void innerCreateEmptyObject(OBSFileSystem owner, String objectName) throws ObsException, IOException {
        InputStream im = new InputStream(){

            @Override
            public int read() {
                return -1;
            }
        };
        PutObjectRequest putObjectRequest = OBSCommonUtils.newPutObjectRequest(owner, objectName, OBSObjectBucketUtils.newObjectMetadata(0L), im);
        long len = putObjectRequest.getFile() != null ? putObjectRequest.getFile().length() : putObjectRequest.getMetadata().getContentLength().longValue();
        try {
            owner.getObsClient().putObject(putObjectRequest);
            owner.getSchemeStatistics().incrementWriteOps(1);
            owner.getSchemeStatistics().incrementBytesWritten(len);
        }
        finally {
            im.close();
        }
    }

    private static void copyFile(OBSFileSystem owner, String srcKey, String dstKey, long size) throws IOException, InterruptedIOException {
        for (int retryTime = 1; retryTime < 3; ++retryTime) {
            try {
                OBSObjectBucketUtils.innerCopyFile(owner, srcKey, dstKey, size);
                return;
            }
            catch (InterruptedIOException e) {
                throw e;
            }
            catch (IOException e) {
                LOG.warn("Failed to copy file from [{}] to [{}] with size [{}], retry time [{}], exception [{}]", new Object[]{srcKey, dstKey, size, retryTime, e});
                try {
                    Thread.sleep(10L);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        OBSObjectBucketUtils.innerCopyFile(owner, srcKey, dstKey, size);
    }

    private static void innerCopyFile(OBSFileSystem owner, String srcKey, String dstKey, long size) throws IOException {
        LOG.debug("copyFile {} -> {} ", (Object)srcKey, (Object)dstKey);
        try {
            if (size > owner.getCopyPartSize()) {
                InitiateMultipartUploadRequest request = new InitiateMultipartUploadRequest(owner.getBucket(), dstKey);
                request.setAcl(owner.getCannedACL());
                if (owner.getSse().isSseCEnable()) {
                    request.setSseCHeader(owner.getSse().getSseCHeader());
                } else if (owner.getSse().isSseKmsEnable()) {
                    request.setSseKmsHeader(owner.getSse().getSseKmsHeader());
                }
                InitiateMultipartUploadResult result = owner.getObsClient().initiateMultipartUpload(request);
                String uploadId = result.getUploadId();
                LOG.debug("Multipart copy file, uploadId: {}", (Object)uploadId);
                long partCount = OBSObjectBucketUtils.calPartCount(owner.getCopyPartSize(), size);
                List<PartEtag> partEtags = OBSObjectBucketUtils.getCopyFilePartEtags(owner, srcKey, dstKey, size, uploadId, partCount);
                CompleteMultipartUploadRequest completeMultipartUploadRequest = new CompleteMultipartUploadRequest(owner.getBucket(), dstKey, uploadId, partEtags);
                owner.getObsClient().completeMultipartUpload(completeMultipartUploadRequest);
            } else {
                ObjectMetadata srcom = OBSObjectBucketUtils.getObjectMetadata(owner, srcKey);
                ObjectMetadata dstom = OBSObjectBucketUtils.cloneObjectMetadata(srcom);
                CopyObjectRequest copyObjectRequest = new CopyObjectRequest(owner.getBucket(), srcKey, owner.getBucket(), dstKey);
                copyObjectRequest.setAcl(owner.getCannedACL());
                copyObjectRequest.setNewObjectMetadata(dstom);
                if (owner.getSse().isSseCEnable()) {
                    copyObjectRequest.setSseCHeader(owner.getSse().getSseCHeader());
                    copyObjectRequest.setSseCHeaderSource(owner.getSse().getSseCHeader());
                } else if (owner.getSse().isSseKmsEnable()) {
                    copyObjectRequest.setSseKmsHeader(owner.getSse().getSseKmsHeader());
                }
                owner.getObsClient().copyObject(copyObjectRequest);
            }
            owner.getSchemeStatistics().incrementWriteOps(1);
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("copyFile(" + srcKey + ", " + dstKey + ")", srcKey, e);
        }
    }

    static int calPartCount(long partSize, long cloudSize) {
        long partCount = cloudSize % partSize == 0L ? cloudSize / partSize : cloudSize / partSize + 1L;
        return (int)partCount;
    }

    static List<PartEtag> getCopyFilePartEtags(OBSFileSystem owner, String srcKey, String dstKey, long objectSize, String uploadId, long partCount) throws IOException {
        List<PartEtag> partEtags = Collections.synchronizedList(new ArrayList());
        ArrayList partCopyFutures = new ArrayList();
        OBSObjectBucketUtils.submitCopyPartTasks(owner, srcKey, dstKey, objectSize, uploadId, partCount, partEtags, partCopyFutures);
        try {
            for (Future future : partCopyFutures) {
                future.get();
            }
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while copying objects (copy)");
            throw new InterruptedIOException("Interrupted while copying objects (copy)");
        }
        catch (ExecutionException e) {
            LOG.error("Multipart copy file exception.", (Throwable)e);
            for (Future future : partCopyFutures) {
                future.cancel(true);
            }
            owner.getObsClient().abortMultipartUpload(new AbortMultipartUploadRequest(owner.getBucket(), dstKey, uploadId));
            throw OBSCommonUtils.extractException("Multi-part copy with id '" + uploadId + "' from " + srcKey + "to " + dstKey, dstKey, e);
        }
        partEtags.sort(Comparator.comparingInt(PartEtag::getPartNumber));
        return partEtags;
    }

    private static void submitCopyPartTasks(OBSFileSystem owner, String srcKey, String dstKey, long objectSize, String uploadId, long partCount, List<PartEtag> partEtags, List<Future<?>> partCopyFutures) {
        int i = 0;
        while ((long)i < partCount) {
            long rangeStart = (long)i * owner.getCopyPartSize();
            long rangeEnd = (long)(i + 1) == partCount ? objectSize - 1L : rangeStart + owner.getCopyPartSize() - 1L;
            int partNumber = i + 1;
            partCopyFutures.add(owner.getBoundedCopyPartThreadPool().submit(() -> {
                CopyPartRequest request = new CopyPartRequest();
                request.setUploadId(uploadId);
                request.setSourceBucketName(owner.getBucket());
                request.setSourceObjectKey(srcKey);
                request.setDestinationBucketName(owner.getBucket());
                request.setDestinationObjectKey(dstKey);
                request.setByteRangeStart(Long.valueOf(rangeStart));
                request.setByteRangeEnd(Long.valueOf(rangeEnd));
                request.setPartNumber(partNumber);
                if (owner.getSse().isSseCEnable()) {
                    request.setSseCHeaderSource(owner.getSse().getSseCHeader());
                    request.setSseCHeaderDestination(owner.getSse().getSseCHeader());
                }
                CopyPartResult result = owner.getObsClient().copyPart(request);
                partEtags.add(new PartEtag(result.getEtag(), Integer.valueOf(result.getPartNumber())));
                LOG.debug("Multipart copy file, uploadId: {}, Part#{} done.", (Object)uploadId, (Object)partNumber);
            }));
            ++i;
        }
    }

    private static ObjectMetadata cloneObjectMetadata(ObjectMetadata source) {
        ObjectMetadata ret = OBSObjectBucketUtils.newObjectMetadata(source.getContentLength());
        if (source.getContentEncoding() != null) {
            ret.setContentEncoding(source.getContentEncoding());
        }
        return ret;
    }

    static OBSFileStatus innerGetObjectStatus(OBSFileSystem owner, Path f) throws IOException {
        String key;
        Path path;
        block11: {
            path = OBSCommonUtils.qualify(owner, f);
            key = OBSCommonUtils.pathToKey(owner, path);
            LOG.debug("Getting path status for {}  ({})", (Object)path, (Object)key);
            if (!StringUtils.isEmpty((CharSequence)key)) {
                try {
                    ObjectMetadata meta = OBSObjectBucketUtils.getObjectMetadata(owner, key);
                    if (OBSCommonUtils.objectRepresentsDirectory(key, meta.getContentLength())) {
                        LOG.debug("Found exact file: fake directory");
                        return new OBSFileStatus(path, owner.getUsername());
                    }
                    LOG.debug("Found exact file: normal file");
                    return new OBSFileStatus(meta.getContentLength(), OBSCommonUtils.dateToLong(meta.getLastModified()), path, owner.getDefaultBlockSize(path), owner.getUsername());
                }
                catch (ObsException e) {
                    if (e.getResponseCode() != 404) {
                        throw OBSCommonUtils.translateException("getFileStatus", path, e);
                    }
                    if (key.endsWith("/")) break block11;
                    String newKey = key + "/";
                    try {
                        ObjectMetadata meta = OBSObjectBucketUtils.getObjectMetadata(owner, newKey);
                        if (OBSCommonUtils.objectRepresentsDirectory(newKey, meta.getContentLength())) {
                            LOG.debug("Found file (with /): fake directory");
                            return new OBSFileStatus(path, owner.getUsername());
                        }
                        LOG.debug("Found file (with /): real file? should not happen: {}", (Object)key);
                        return new OBSFileStatus(meta.getContentLength(), OBSCommonUtils.dateToLong(meta.getLastModified()), path, owner.getDefaultBlockSize(path), owner.getUsername());
                    }
                    catch (ObsException e2) {
                        if (e2.getResponseCode() == 404) break block11;
                        throw OBSCommonUtils.translateException("getFileStatus", newKey, e2);
                    }
                }
            }
        }
        try {
            boolean isEmpty = OBSCommonUtils.innerIsFolderEmpty(owner, key);
            LOG.debug("Is dir ({}) empty? {}", (Object)path, (Object)isEmpty);
            return new OBSFileStatus(path, owner.getUsername());
        }
        catch (ObsException e) {
            if (e.getResponseCode() != 404) {
                throw OBSCommonUtils.translateException("getFileStatus", key, e);
            }
            LOG.debug("Not Found: {}", (Object)path);
            throw new FileNotFoundException("No such file or directory: " + path);
        }
    }

    static ContentSummary getDirectoryContentSummary(OBSFileSystem owner, String key) throws IOException {
        String newKey = key;
        newKey = OBSCommonUtils.maybeAddTrailingSlash(newKey);
        long[] summary = new long[]{0L, 0L, 1L};
        LOG.debug("Summary key {}", (Object)newKey);
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(owner.getBucket());
        request.setPrefix(newKey);
        TreeSet<String> directories = new TreeSet<String>();
        request.setMaxKeys(owner.getMaxKeys());
        ObjectListing objects = OBSCommonUtils.listObjects(owner, request);
        while (true) {
            if (!objects.getCommonPrefixes().isEmpty() || !objects.getObjects().isEmpty()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Found path as directory (with /): {}/{}", (Object)objects.getCommonPrefixes().size(), (Object)objects.getObjects().size());
                }
                for (String prefix : objects.getCommonPrefixes()) {
                    LOG.debug("Objects in folder [" + prefix + "]:");
                    OBSObjectBucketUtils.getDirectories(prefix, newKey, directories);
                }
                for (ObsObject obj : objects.getObjects()) {
                    LOG.debug("Summary: {} {}", (Object)obj.getObjectKey(), (Object)obj.getMetadata().getContentLength());
                    if (!obj.getObjectKey().endsWith("/")) {
                        summary[0] = summary[0] + obj.getMetadata().getContentLength();
                        summary[1] = summary[1] + 1L;
                    }
                    OBSObjectBucketUtils.getDirectories(obj.getObjectKey(), newKey, directories);
                }
            }
            if (!objects.isTruncated()) break;
            objects = OBSCommonUtils.continueListObjects(owner, objects);
        }
        summary[2] = summary[2] + (long)directories.size();
        LOG.debug(String.format("file size [%d] - file count [%d] - directory count [%d] - file path [%s]", summary[0], summary[1], summary[2], newKey));
        return new ContentSummary.Builder().length(summary[0]).fileCount(summary[1]).directoryCount(summary[2]).spaceConsumed(summary[0]).build();
    }

    private static void getDirectories(String key, String sourceKey, Set<String> directories) {
        Optional parent;
        Path p = new Path(key);
        Path sourcePath = new Path(sourceKey);
        if (key.endsWith("/") && p.compareTo(sourcePath) > 0) {
            directories.add(p.toString());
        }
        while (p.compareTo(sourcePath) > 0 && (parent = p.getOptionalParentPath()).isPresent() && (p = (Path)parent.get()).compareTo(sourcePath) != 0) {
            directories.add(p.toString());
        }
    }

    private static Future<CopyObjectResult> copyFileAsync(OBSFileSystem owner, String srcKey, String dstKey, long size) {
        return owner.getBoundedCopyThreadPool().submit(() -> {
            OBSObjectBucketUtils.copyFile(owner, srcKey, dstKey, size);
            return null;
        });
    }
}

