/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model.system;

import com.datastax.bdp.graph.api.DseGraph;

/**
 * Builder pattern for creating new graph instances in DSE Graph. Invoked via {@link GraphSystem#createGraph(String)}
 * where the name of the graph instance is specified.
 *
 * The builder allows the user to pre-set configuration options.
 *
 * @author Matthias Broecheler (me@matthiasb.com)
 */
public interface GraphBuilder
{

    /**
     * When invoked, the builder checks whether a graph of the given name exists and when it does it
     * returns that instance rather than attempting to open a new one. If an existing graph is returned
     * in {@link #build()} any configuration options set through this builder are ignored.
     *
     * If this method is not invoked and a graph of the same name already exists, an exception is thrown.
     *
     * @return This builder instance
     */
    GraphBuilder ifNotExists();

    /**
     * Sets the provided value for the given configuration option on the graph instance that is created
     * through this builder.
     *
     * The option may either be an option recognized by DSE Graph or Tinkerpop.
     *
     * @param configOption The name of the option
     * @param value The value of the option
     * @return This builder instance
     */
    GraphBuilder set(String configOption, Object value);

    /**
     * Builds the graph instance and sets the configuration options.
     *
     * @return A new or existing graph instance
     * @throws IllegalArgumentException If a graph of the provided name already exists and {@link #ifNotExists()}
     *          has not been invoked.
     */
    DseGraph build();

}
