/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.system;

import com.datastax.bdp.graph.api.DseGraph;
import com.datastax.bdp.graph.api.model.system.GraphBuilder;
import com.datastax.bdp.graph.api.model.system.GraphSystem;
import com.datastax.bdp.graph.api.schema.SchemaImpl;
import com.github.misberner.duzzt.annotations.GenerateEmbeddedDSL;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;

/**
 *
 *
 * @author Matthias Broecheler (me@matthiasb.com)
 */
@GenerateEmbeddedDSL(
        name = "GraphBuilder",
        syntax = "(replication? systemReplication? (option set)* ifNotExists? create) | ifExists? drop | ifExists? truncate | describe | exists"
)
public class GraphBuilderImpl {

    public static final String REPLICATION_CONFIG = "replication_config";
    public static final String SYSTEM_REPLICATION_CONFIG = "system_replication_config";
    private final GraphSystem system;
    private final String name;
    private boolean ifExists;

    public GraphBuilderImpl(GraphSystem system, String name) {
        this.system = system;
        this.name = name;
    }

    public void drop() {
        if(!ifExists || system.graphExists(name)) {
            system.dropGraph(name);
        }
    }

    public void truncate() {
        if(!ifExists || system.graphExists(name))
        {
            system.truncateGraph(name);
        }
    }

    public String describe() {
        return describe(name);
    }

    static String describe(String graphName) {
        return "system.graph(" + SchemaImpl.quote(graphName) + ").create()";
    }

    public boolean exists() {
        return system.graphExists(name);
    }

    private GraphBuilder builder = null;
    private String configOption = null;

    private final GraphBuilder builder() {
        if (builder == null) builder = system.createGraph(name);
        return builder;
    }

    public void option(String configOption) {
        if (StringUtils.isBlank(configOption)) throw new IllegalArgumentException("Invalid config option");
        this.configOption = configOption;
    }

    public void set(Object value) {
        Preconditions.checkArgument(configOption!=null,"Invalid invocation - need to specify config option first");
        builder().set(configOption,value);
        configOption = null;
    }

    public void ifExists() {
        ifExists = true;
    }

    public void ifNotExists() {
        builder().ifNotExists();
    }

    public void create() {
        builder().build();
    }

    public void replication(String replication) {
        builder().set(REPLICATION_CONFIG, replication);
    }

    public void systemReplication(String replication) {
        builder().set(SYSTEM_REPLICATION_CONFIG, replication);
    }

}
