/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.snapshot;

import java.util.concurrent.ExecutionException;

import org.apache.tinkerpop.gremlin.process.traversal.dsl.graph.GraphTraversalSource;

/**
 * Builds a persistent analytics-capable graph data snapshot.
 */
public interface SnapshotBuilder
{
    /**
     * Set the vertex labels to retain in the snapshot.
     * <p>
     * If this method is never called on a {@linkplain SnapshotBuilder},
     * then it retains all vertices regardless of label.
     *
     * @param labels one or more vertex label names to snapshot
     * @return this builder
     */
    SnapshotBuilder vertices(String... labels);

    /**
     * Set the edge labels to retain in the snapshot.
     * <p>
     * If this method is never called on a {@linkplain SnapshotBuilder},
     * then it retains all edges regardless of label.
     *
     * @param labels one or more edge label names to snapshot
     * @return this builder
     */
    SnapshotBuilder edges(String... labels);

    /**
     * Set or override an arbitrary key-value string pair in the
     * configuration of the Spark job used to create the snapshot.
     *
     * @param key the configuration option to modify
     * @param value the configuraiton value
     * @return this builder
     */
    SnapshotBuilder conf(String key, String value);

    /**
     * Make the snapshot.  Though the details are implementation dependent, callers
     * should assume that this is a resource-intensive batch operation.
     *
     * @return a traversal source attached to the just-created snapshot
     */
    GraphTraversalSource create() throws InterruptedException, ExecutionException;
}
