/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model;

import java.util.Set;

import com.datastax.bdp.graph.api.Identified;
import com.datastax.bdp.graph.api.Named;

/**
 * Vertex indexes allow quick retrieval of vertices by particular property keys through gremlin queries.
 */
public interface VertexIndex extends Named, Identified
{

    /**
     * DSE Graph supports the following types of indexes which are distinguished by their underlying index structures.
     */
    enum Type
    {
        /**
         * The index is build using materialized views. This type of index should only be used for highly selective indexes
         * and equality condition retrievals. This is the fastest type of index
         */
        Materialized,

        /**
         * This index is build using secondary indexes. It only supports equality conditions. It is slower than {@link #Materialized}
         * but can index properties with lower selectivity.
         */
        Secondary,

        /**
         * This index uses Solr indexing. It supports a wide spectrum of query conditions and selectivities.
         */
        Search
    }

    /**
     * @return The {@link PropertyKey}'s by which the vertex is indexed
     */
    Set<? extends IndexedPropertyKey> propertyKeys();

    /**
     * @return the type of this index
     */
    Type getType();


    /**
     * Whether this index is a unique index, i.e. it should only return at most one result per index hit.
     *
     * @return
     */
    boolean isUnique();

    /**
     * @return The vertex label that is indexed
     */
    VertexLabel vertexLabel();

    /**
     * Drops the index
     */
    void drop();

    /**
     * Drops the property key from this index. Only supported for {@link Type#Search} indexes.
     *
     * @param propertyKey Drop the property key from this index
     */
    void dropPropertyKey(String propertyKey);

    interface Builder {

        /**
         * Sets the {@link VertexIndex.Type} of this index. The type must be set.
         *
         * @param type The type of this index
         * @return This builder.
         */
        Builder type(VertexIndex.Type type);

        /**
         * Sets the {@link VertexIndex.Type} of this index to {@link Type#Search}
         *
         * @return This builder.
         * @see #type(VertexIndex.Type)
         */
        default Builder search() {
            return type(Type.Search);
        }

        /**
         * Sets the {@link VertexIndex.Type} of this index to {@link Type#Materialized}
         *
         * @return This builder.
         * @see #type(VertexIndex.Type)
         */
        default Builder materialized() {
            return type(Type.Materialized);
        }

        /**
         * Sets the {@link Type} of this index to {@link Type#Secondary}
         *
         * @return This builder.
         * @see #type(VertexIndex.Type)
         */
        default Builder secondary() {
            return type(Type.Secondary);
        }

        /**
         * Indexes a property key using default options.
         *
         * On Search indexes, this method's behavior depends on whether the {@code propertyKey}
         * is part of the vertex label's ID.  If it is part of the vertex label's ID, then
         * this method indexes the property key as a plain string, without fulltext analysis.
         * If it is not part of the vertex label's ID, then this method indexes the property
         * key as both a plain string and with fulltext analysis.  To force the latter
         * behavior on properties that are part of the vertex label's ID, use
         * {@link #byPropertyKeyAsStringAndText(String)} instead.
         *
         * @param propertyKey The property key by which vertices are indexed
         * @return This builder.
         */
        Builder byPropertyKey(String propertyKey);

        /**
         * Only supported for {@link Type#Search} indexes
         *
         * @param propertyKey The property key by which vertices are indexed
         * @param option The text index option for configuring the indexing mechanism.
         * @return This builder.
         */
        Builder byPropertyKeyAsText(String propertyKey, IndexOption.Text option);

        /**
         * Force indexing as both plain strings and analyzed text for the given property.
         *
         * Only supported for {@link Type#Search} indexes.
         *
         * @param propertyKey the property key by which vertices are indexed
         * @return this builder
         */
        Builder byPropertyKeyAsStringAndText(String propertyKey);

        /**
         * Only supported for {@link Type#Search} geo indexes
         *
         * @param propertyKey The property key by which vertices are indexed
         * @return
         */
        Builder byPropertyKeyWithError(String propertyKey, Double maxDistErr, Double distErrPct);

        /**
         * @return if the vertex index already exists then just return it.
         */
        Builder ifNotExists();

        /**
         * This is not yet supported
         *
         * @return Indicates that the index may only return a unique result.
         */
        Builder unique();

        /**
         * @return Add the index.
         */
        VertexIndex add();

    }


    /**
     * Options that determine how properties are being indexed
     */
    interface IndexOption {

        String describe();

        boolean isDescriptionSuppressed();

        enum Text {

            FULLTEXT, STRING

        }

    }
}
