/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.tinkerpop.gremlin.structure.Direction;

import com.datastax.bdp.graph.api.Identified;
import com.datastax.bdp.graph.api.Named;

/**
 * Represents a type of vertex within the graph.
 */
public interface VertexLabel extends Named, Identified
{

    /**
     * Retrieve an index associated with this vertex label.
     * @param name The name of the index.
     * @return The index or null if no such index exists.
     */
    VertexIndex vertexIndex(String name);

    /**
     * @param name The name of the edge index
     * @return The edge index or null if it is doesn't exist.
     */
    EdgeIndex edgeIndex(String name);


    /**
     * @param name The name of the property index.
     * @return The property index or null if it doesn't exist.
     */
    PropertyIndex propertyIndex(String name);


    /**
     * Build an index of this vertex label.
     * @param name The name of the index.
     * @return The builder for the index. Note that add() must be called for the index to be
     * added.
     */
    VertexIndex.Builder buildVertexIndex(String name);

    /**
     * Build an edge index for this vertex label.
     * @param name The name of the index.
     * @param label The edge label to be indexed.
     * @return The builder for the index. Note that {@link EdgeIndex.Builder#add()} must be called for the index to be
     * added.
     */
    EdgeIndex.Builder buildEdgeIndex(String name, String label);


    /**
     * Build a property index for this vertex label.
     * @param name The name of the index.
     * @param propertyKey The propertyKey to be indexed.
     * @return The builder for the index. Note that {@link PropertyIndex.Builder#add()} must be called for the index to be
     * added.
     */
    PropertyIndex.Builder buildPropertyIndex(String name, String propertyKey);

    /**
     * @return All the indices associated with this vertex label.
     */
    Set<? extends VertexIndex> vertexIndices();

    /**
     * @return All the indices associated with this vertex label.
     */
    Set<? extends EdgeIndex> edgeIndices();

    /**
     * @return All the indices associated with this relation type.
     */
    Set<? extends PropertyIndex> propertyIndices();



    /**
     * @return the property keys associated with this vertex label.
     */
    Set<? extends PropertyKey> propertyKeys();


    /**
     * @return All the edge labels that are associated with this vertex label.
     */
    Set<? extends EdgeLabel> edgeLabels();

    /**
     * @return The columns that are used to make up the ID.
     */
    Set<? extends IdPropertyKey> idPropertyKeys();


    /**
     * @return The time to live of vertices of this type.
     */
    Optional<Duration> ttl();


    /**
     * Returns all {@link CacheConfig} cache configurations for this vertex label.
     *
     * @return
     */
    Set<? extends CacheConfig> cacheConfigs();

    /**
     * Configures the global property cache time for this vertex label. The given duration configures the maximum time that
     * property result sets may be cached locally.
     *
     * Providing {@link Duration#ZERO} as the argument for {@code cacheTime} disables property caching.
     *
     * @param cacheTime The maximum time result sets can be cached locally, {@link Duration#ZERO} to disable.
     */
    void setPropertyCacheTime(Duration cacheTime);


    /**
     * Configures the edge label cache time for this vertex label. The given duration configures the maximum time that
     * adjacency result sets for these edge labels may be cached locally.
     *
     * Providing no argument for {@code edgeLabels} configures the general edge cache time which applies to all edge labels.
     *
     * Providing {@link Duration#ZERO} as the argument for {@code cacheTime} disables property caching.
     *
     * @param cacheTime The maximum time result sets can be cached locally, {@link Duration#ZERO} to disable.
     */
    void setEdgeCacheTime(Duration cacheTime, String... edgeLabels);



    /**
     * @return The set of known adjacencies to this vertex label.
     */
    Set<? extends Adjacency> adjacencies();

    /**
     * Register that vertices with this label may be connected to another label via edge.
     * @param edgeLabel The edge label to register.
     * @param direction The direction of use, IN, OUT or BOTH.
     * @param adjacentVertexLabel The adjacent label on the other end of the edge.
     */
    void addAdjacency(String edgeLabel, Direction direction, String adjacentVertexLabel);


    /**
     * Add an adjacency to the vertex label in both directions
     *
     * @param edgeLabel The name of the edge label
     * @param adjacentVertexLabel The name of adjacent vertex label
     */
    default void addAdjacency(String edgeLabel, String adjacentVertexLabel) {
        addAdjacency(edgeLabel, Direction.BOTH, adjacentVertexLabel);
    }

    /**
     * Register that a particular property key may be used on vertices with this label.
     * @param name The property key name
     */
    PropertyKey addPropertyKey(String name);

    /**
     *
     * @return Whether this vertex label uses standard ids
     */
    boolean hasStandardId();

    /**
     * Drop the vertex label from the graph
     */
    void drop();

    /**
     * Drops the property key from this vertex label
     *
     * @param name The property key to drop
     */
    void dropPropertyKey(String name);

    interface Builder {


        /**
         * Create and add the vertex label to the schema.
         *
         * @return The added vertex label.
         */
        VertexLabel add();


        /**
         * @param ttl The time to live for the vertex label. Once expired the vertex will be automatically removed.
         *
         * @return This builder.
         */
        Builder ttl(Duration ttl);


        /**
         * Add a key that makes up the ID component of this vertex label.
         * Id properties must be set upon vertex construction and cannot be changed later.
         * If no ID components are added then the default ID components will be used.
         *
         * If custom ids are used then at least one component of type {@link IdPropertyKey.Type#Partition} is required.
         *
         * @param propertyKey The name of the property key. This key already have been defined.
         * @param type The type of the id component as defined in {@link com.datastax.bdp.graph.api.model.IdPropertyKey.Type}.
         * @return This builder.
         */
        Builder idComponent(String propertyKey, IdPropertyKey.Type type);

        /**
         * Adds the given property key as a partition ID to this vertex labels custom vertex id.
         *
         * @param propertyKey The name of the property key. This key already have been defined.
         * @return This builder.
         * @see #idComponent(String, IdPropertyKey.Type)
         */
        default Builder partitionId(String propertyKey) {
            return idComponent(propertyKey, IdPropertyKey.Type.Partition);
        }

        /**
         * Adds the given property key as a clustering ID to this vertex labels custom vertex id.
         *
         * @param propertyKey The name of the property key. This key already have been defined.
         * @return This builder.
         */
        default Builder clusteringId(String propertyKey) {
            return idComponent(propertyKey, IdPropertyKey.Type.Clustering);
        }


        /**
         * Add a key that makes up the id component of this vertex label.
         *
         * @param propertyKey The property key.
         * @param type The type of the id component.
         * @return This builder.
         * @see #idComponent(String, IdPropertyKey.Type)
         */
        Builder idComponent(PropertyKey propertyKey, IdPropertyKey.Type type);

        /**
         * Add property keys to the vertex label
         *
         * @param propertyKeys The property key.
         */
        Builder addPropertyKeys(String... propertyKeys);

        /**
         * Add an adjacency to the vertex label
         *
         * @param edgeLabel The name of the edge label
         * @param direction The direction of the allowed edge. May be IN, OUT or BOTH
         * @param adjacentVertexLabel The name of adjacent vertex label
         */
        Builder addAdjacency(String edgeLabel, Direction direction, String adjacentVertexLabel);


        /**
         * Add an adjacency to the vertex label in both directions
         *
         * @param edgeLabel The name of the edge label
         * @param adjacentVertexLabel The name of adjacent vertex label
         */
        default Builder addAdjacency(String edgeLabel, String adjacentVertexLabel) {
            return addAdjacency(edgeLabel, Direction.BOTH, adjacentVertexLabel);
        }

        /**
         * @return if the vertex label already exists then just return it.
         */
        Builder ifNotExists();

    }


}
