/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api;

import com.datastax.bdp.graph.api.snapshot.SnapshotBuilder;
import org.apache.tinkerpop.gremlin.structure.Graph;

import com.datastax.bdp.graph.api.schema.Schema;

/**
 * A Datastax Graph.
 * Allows access to config and schema.
 */
public interface DseGraph extends Graph
{

    /**
     * The key that represents the graph name that must be specified in the configuration.
     */
    String NAME = "gremlin.graph.name";

    /**
     * The user name for the connection
     */
    String USER = "gremlin.graph.user";


    /**
     * Returns the name of this graph which uniquely identifies this graph within the cluster.
     *
     * @return the name of this graph database
     */
    String name();

    /**
     * Checks whether the graph is open.
     *
     * @return true, if the graph is open, else false.
     * @see #close()
     */
    boolean isOpen();

    /**
     * Returns a schema handle for the graph schema. This handle can be used to inspect the schema as well as modify it.
     * The schema handle is tied to the current transactional context. In particular, any changes to the schema will only
     * be migrated with the enclosing transaction commits.
     *
     * @return The schema of the graph.
     */
    Schema schema();

    /**
     * Returns a builder that makes durable snapshots of specific vertex and edge labels.
     *
     * @return a new snapshot builder
     */
    SnapshotBuilder snapshot();

    /**
     * Returns the allow_scan option value for this graph's internal, automatically-managed transaction.
     * {@link Schema#getEffectiveAllowScan()} is preferred over this method.  This method is only necessary
     * when the user overrides the allow_scan value at the transaction level.
     */
    boolean getEffectiveAllowScan();

    class Exceptions {

        public static IllegalStateException graphIsClosed() {
            return new IllegalStateException("Graph is closed");
        }

        public static UnsupportedOperationException graphComputerInTx() {
            return new UnsupportedOperationException("GraphComputer is only supported on the graph and not an individual transaction");
        }

    }

}
