/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model;

import java.time.Duration;
import java.util.Optional;
import java.util.Set;

import com.datastax.bdp.graph.api.Identified;
import com.datastax.bdp.graph.api.Named;

/**
 * Represents a type of edge within the graph.
 */
public interface EdgeLabel extends Named, Identified
{

     /**
     * The directionality of the edge.
     */
    enum Directionality {
        /**
         * Edges are traversable in both directions.
         */
        Bidirectional,
        /**
         * Edges are traversable in only one direction. (CURRENTLY UNSUPPORTED)
         */
        Unidirectional

    }

    /**
     * @return The directionality of the edge. Only bidirectional is supported.
     */
    Directionality directionality();

    /**
     * @return The time to live for the relation type. Once expired the relation will be automatically removed.
     */
    Optional<Duration> ttl();

    /**
     * @return The cardinality of this edge label.
     */
    Cardinality cardinality();

    /**
     * @return All property keys associated with this edge label.
     */
    Set<? extends PropertyKey> propertyKeys();

    /**
     * Register that a particular property key may be used on edges with this label.
     * @param name The property key name
     */
    PropertyKey addPropertyKey(String name);

    /**
     * Drop the edge label from the graph. The data is not actually removed.
     */
    void drop();

    /**
     * Drops the property key from this vertex label
     *
     * @param name The property key to drop
     */
    void dropPropertyKey(String name);

    interface Builder {

        /**
         * Sets the cardinality for this edge label. If the cardinality is set to {@link Cardinality#Single} then only
         * a single edge of this label may exists between any pair of vertices in a given direction. If the cardinality
         * is {@link Cardinality#Multiple} then multiple edges can exist.
         *
         * The default cardinality is {@link Cardinality#Multiple}.
         *
         * @param cardinality The cardinality of this edge label.
         * @return This builder.
         */
        Builder cardinality(Cardinality cardinality);

        /**
         * Sets the cardinality for this edge label to single.
         *
         * @return This builder
         * @see #cardinality(Cardinality)
         */
        default Builder single() {
            return cardinality(Cardinality.Single);
        }

        /**
         * Sets the cardinality for this edge label to multiple (which is the default).
         *
         * @return This builder
         * @see #cardinality(Cardinality)
         */
        default Builder multiple() {
            return cardinality(Cardinality.Multiple);
        }

        /**
         * Specifies the edge to be {@link Directionality#Unidirectional} or
         * {@link Directionality#Bidirectional}. Unidirectional edges
         * can only be traversed in one direction making them suitable for linking to super nodes. The default
         * is {@link Directionality#Bidirectional}
         * @param directionality The directionality of the edge.
         * @return the builder
         * TODO Directionality unsupported for now.
         */
       // Builder directionality(Directionality directionality);

        /**
         * @param ttl The time to live for the edge label. Once expired the edge will be automatically removed.
         * @return This builder.
         */
        Builder ttl(Duration ttl);

        /**
         * @return if the edge label already exists then just return it.
         */
        Builder ifNotExists();


        /**
         * Create and add the edge label to the schema.
         * @return the added edge label.
         */
        EdgeLabel add();


        /**
         * Add property keys to the edge label
         *
         * @param propertyKeys The property key.
         */
        Builder addPropertyKeys(String... propertyKeys);

    }
}
