/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model;


import java.util.Collection;

import org.apache.tinkerpop.gremlin.process.traversal.dsl.graph.GraphTraversalSource;
import com.datastax.bdp.graph.api.model.config.GraphConfig;

/**
 * Interface for defining the schema of a graph.
 * Schema changes are only visible to the current transaction until the transaction is committed.
 */
public interface Schema
{

    /**
     * The schema mode dictates how the schema may be changed and allows the user to control the tradeoff between
     * flexibility and safety.
     */
    enum Mode {
        /**
         * Properties, vertex labels and edge labels are created automatically when they are first used.
         */
        Development,

        /**
         * Properties, vertex labels and edge labels must be defined before use.
         * Properties must added to vertex and edge labels before their first use.
         * Adjacency relationships between vertex labels must be added before first use.
         */
        Production,

        /**
         * Uses the system default
         */
        Default;
    }

    /**
     * Retrieve a property key by name.
     * @param name The name of the property key.
     * @return The property key or null if it is not present.
     */
    PropertyKey propertyKey(String name);

    /**
     * Retrieve a vertex label by name.
     * @param name The name of the vertex label.
     * @return The vertex label or null if it is not present.
     */
    VertexLabel vertexLabel(String name);

    /**
     * Retrieve an edge label by name.
     * @param name The name of the edge label.
     * @return The edge label or null if it is not present.
     */
    EdgeLabel edgeLabel(String name);

    /**
     * @return All vertex labels defined.
     */
    Collection<? extends VertexLabel> vertexLabels();


    /**
     * @return All edge labels defined.
     */
    Collection<? extends EdgeLabel> edgeLabels();

    /**
     * @return All property keys defined.
     */
    Collection<? extends PropertyKey> propertyKeys();

    /**
     * @return The mode of this schema.
     */
    Mode mode();


    /**
     * Build a property key that may be used on vertex labels or edge labels.
     * @param name The name of the property key.
     * @param type The data type of the property's value.
     * @return The builder for the property. Note that {@link PropertyKey.Builder#add()} must be called
     * for the property to be added.
     */
    PropertyKey.Builder buildPropertyKey(String name, ValueType type);

    /**
     * Build a property key that may be used on vertex labels or edge labels.
     * @param name The name of the property key.
     * @param type The data type of the property's value.
     * @return The builder for the property. Note that {@link PropertyKey.Builder#add()} must be called
     * for the property to be added.
     */
    PropertyKey.Builder buildPropertyKey(String name, Class<?> type);


    /**
     * Build a vertex label.
     * @param name The name of the vertex label.
     * @return The builder for the vertex label. Note that {@link VertexLabel.Builder#add()} must be called
     * for the vertex label to be added.
     */
    VertexLabel.Builder buildVertexLabel(String name);

    /**
     * Build an edge label.
     * @param name The name of the edge label.
     * @return The builder for the edge label. Note that {@link EdgeLabel.Builder#add()} must be called
     * for the edge label to be added.
     */
    EdgeLabel.Builder buildEdgeLabel(String name);


    /**
     * Drop all labels and property keys.
     */
    void drop();


    /**
     * Start a traversal from the graph representation of this schema.
     * @return The graph traversal.
     */
    GraphTraversalSource traversal();

    /**
     *
     * @return The graph configuration for the graph associated with this schema. A schema commit triggers config changes
     * to be persisted.
     */
    GraphConfig configuration();

    /**
     * Get the effective value of the schema_mode option, considering graph-level configuration,
     * dse.yaml configuration, and the default, in that order.
     */
    String getEffectiveSchemaMode();

    /**
     * Get the effective value of the allow_scan option, considering graph-level configuration,
     * the effective schema_mode value, and the default, in that order.
     * This method does not consider any transaction-level override that the user may have
     * applied to the allow_scan option.
     */
    boolean getEffectiveAllowScan();

    boolean isStandardIdKey(PropertyKey key);

}
