/*
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */

package com.datastax.bdp.graph.api;

import java.lang.reflect.Modifier;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.MonthDay;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.Period;
import java.time.Year;
import java.time.YearMonth;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.datastax.bdp.graph.api.datatypes.Tuple;
import com.datastax.bdp.graph.api.datatypes.UDT;
import com.datastax.bdp.graph.api.id.DsegId;
import com.datastax.bdp.graph.api.id.SchemaId;
import com.datastax.bdp.graph.api.model.system.GraphBuilder;
import com.datastax.bdp.graph.api.model.system.GraphSystem;
import com.datastax.bdp.graph.api.schema.Configuration;
import com.datastax.bdp.graph.api.schema.EdgeLabel;
import com.datastax.bdp.graph.api.schema.PropertyKey;
import com.datastax.bdp.graph.api.schema.Schema;
import com.datastax.bdp.graph.api.schema.VertexLabel;
import com.datastax.driver.dse.geometry.Distance;
import com.datastax.driver.dse.geometry.Geometry;
import com.datastax.driver.dse.geometry.LineString;
import com.datastax.driver.dse.geometry.Point;
import com.datastax.driver.dse.geometry.Polygon;
import com.datastax.dse.graph.api.predicates.Geo;
import com.datastax.dse.graph.api.predicates.Search;
import org.apache.tinkerpop.gremlin.jsr223.AbstractGremlinPlugin;
import org.apache.tinkerpop.gremlin.jsr223.DefaultImportCustomizer;
import org.apache.tinkerpop.gremlin.jsr223.ImportCustomizer;

/**
 * DSE Graph Plugin that handles imports for the server's ScriptEngine.
 */
public class DseGraphGremlinPlugin extends AbstractGremlinPlugin
{
    private static final String NAME = "datastax.graph";

    private static final ImportCustomizer imports = DefaultImportCustomizer.build()
            .addClassImports(
                    DseGraph.class,                     // com.datastax.bdp.graph.api
                    Identified.class,
                    Named.class,
                    DsegId.class,                       // com.datastax.bdp.graph.api.id
                    SchemaId.class,
                    Geo.class,                          // com.datastax.bdp.graph.api.predicates
                    Search.class,
                    Configuration.class,                // com.datastax.bdp.graph.api.schema
                    EdgeLabel.class,
                    PropertyKey.class,
                    Schema.class,
                    VertexLabel.class,
                    GraphSystem.class,                  // com.datastax.bdp.graph.api.system
                    GraphBuilder.class,
                    Distance.class,                     // com.datastax.driver.dse.geometry
                    Geometry.class,
                    LineString.class,
                    Point.class,
                    Polygon.class,
                    Clock.class  ,                      // java.time
                    Duration.class,
                    InetAddress.class,
                    Instant.class,
                    LocalDate.class,
                    LocalDateTime.class,
                    LocalTime.class,
                    MonthDay.class,
                    OffsetDateTime.class,
                    OffsetTime.class,
                    Period.class,
                    Year.class,
                    YearMonth.class,
                    ZonedDateTime.class,
                    ZoneId.class,
                    ZoneOffset.class,
                    ByteBuffer.class,
                    Tuple.class,
                    UDT.class)
            .addMethodImports(Stream.of(Geo.class.getDeclaredMethods()).
                    filter(m -> Modifier.isStatic(m.getModifiers()) && Modifier.isPublic(m.getModifiers())).
                    collect(Collectors.toList()))
            .addMethodImports(Stream.of(Search.class.getDeclaredMethods()).
                    filter(m -> Modifier.isStatic(m.getModifiers()) && Modifier.isPublic(m.getModifiers())).
                    collect(Collectors.toList()))
            .addEnumImports(Arrays.asList(Geo.Unit.values()))
            .create();

    private static final DseGraphGremlinPlugin instance = new DseGraphGremlinPlugin();

    public DseGraphGremlinPlugin() {
        super(NAME, imports);
    }

    public static DseGraphGremlinPlugin instance() {
        return instance;
    }
}
