/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model.config;

import java.util.Map;

/**
 * A namespace of configuration options within the graph {@link GraphConfig}. Contains configuration options which
 * can be inspected and changed.
 *
 * Iterating over a {@link GraphConfigspace} returns all the configuration entries (i.e. configuration option identifiers
 * and their associated value) that have been set within the space.
 *
 * @author Matthias Broecheler (me@matthiasb.com)
 */
public interface GraphConfigspace extends Iterable<Map.Entry<String,Object>> {

    /**
     *
     * @param configOption The string identifier of a configuration option within this Configspace
     * @return True if the configuration contains the given configuration option
     */
    default boolean contains(String configOption) {
        return get(configOption)!=null;
    }

    /**
     *
     * @param configOption The string identifier of a configuration option within this Configspace
     * @return The value of the configuration option if it has been set, else NULL
     */
    Object get(String configOption);

    /**
     * Sets the given configuration option to the provided value.
     *
     * This changes the configuration but changes will not be applied until {@link GraphConfig#applyChanges()} is called.
     * Make sure to call this method after all configuration changes have been made.     *
     *
     * @param configOption The string identifier of a configuration option within this Configspace
     * @param value The value to set the configuration option to
     * @return This configuration space
     * @throws IllegalArgumentException If the provided value is invalid for the given option
     */
    GraphConfigspace set(String configOption, Object value);

    /**
     * Removes the given configuration option (and its associated value - if any) from this configuration space.
     *
     * @param configOption the name of the option
     * @return This configuration space
     */
    default GraphConfigspace unset(String configOption) {
        return set(configOption,null);
    }

    /**
     *
     * @param namespace A name that identifies a subspace
     * @return the subspace of this {@link GraphConfigspace} as identified by the provided namespace.
     */
//    Configspace subspace(String namespace);

}
