/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.model.system;

import java.util.Set;

import com.datastax.bdp.graph.api.DseGraph;

/**
 * User facing interface that provides information on the graph system and all the graph instances that are available
 * and provides methods to add and remove graphs in the cluster.
 *
 *
 * @author Matthias Broecheler (me@matthiasb.com)
 */
public interface GraphSystem
{

    /**
     * Get all graph names in the cluster.
     *
     * @return The names of all graphs in the cluster
     */
    Set<String> getGraphs();

    /**
     * Check if a graph exists.
     *
     * @param graphName The name of the graph to check
     * @return TRUE, if a graph with the given name exists, else FALSE
     */
    boolean graphExists(String graphName);

    /**
     * Creates a new graph instance
     *
     * @param graphName The name of the graph to be created
     * @return A {@link GraphBuilder} to construct the graph
     */
    GraphBuilder createGraph(String graphName);

    /**
     * Removes the graph of the specified name (and all its associated data as well as meta-data) from
     * the cluster.
     *
     * THIS DATA CANNOT BE RECOVERED!!
     *
     * @param graphName The name of the graph to drop
     */
    void dropGraph(String graphName);

    /**
     * Removes all vertex, property and edge data belonging to the specified graph (meta-data is not affected) from
     * the cluster.
     *
     * THIS DATA CANNOT BE RECOVERED!!
     *
     * @param graphName The name of the graph to truncate
     */
    void truncateGraph(String graphName);



}
