/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.schema;

import java.util.stream.Stream;

import com.datastax.bdp.graph.api.Named;
import com.datastax.bdp.graph.api.model.Schema;
import com.github.misberner.duzzt.annotations.DSLAction;
import com.github.misberner.duzzt.annotations.GenerateEmbeddedDSL;
import org.apache.tinkerpop.gremlin.process.traversal.dsl.graph.GraphTraversalSource;

import static java.util.stream.Collectors.joining;
import static java.util.stream.Stream.concat;

/**
 * TODO
 */

@GenerateEmbeddedDSL(
        name = "Schema",
        syntax = "propertyKey|edgeLabel|vertexLabel|describe|drop|clear|traversal|config|getEffectiveSchemaMode|getEffectiveAllowScan")
public class SchemaImpl
{
    public static final String QM = "\"";
    public static final String NL = "\n";

    private static final String SCHEMA_PREFIX = "schema.";
    public static final String EDGE_LABEL_PREFIX = SCHEMA_PREFIX + "edgeLabel(";
    public static final String PROPERTY_KEY_PREFIX = SCHEMA_PREFIX + "propertyKey(";
    public static final String VERTEX_LABEL_PREFIX = SCHEMA_PREFIX + "vertexLabel(";

    private final Schema model;

    public SchemaImpl(com.datastax.bdp.graph.api.model.Schema model) {
        this.model = model;
    }

    @DSLAction(displayed = true)
    VertexLabel vertexLabel(String name)
    {
        return new VertexLabel(model, name);
    }

    @DSLAction(displayed = true)
    EdgeLabel edgeLabel(String name)
    {
        return new EdgeLabel(model, name);
    }

    @DSLAction(displayed = true)
    PropertyKey propertyKey(String name)
    {
        return new PropertyKey(model, name);
    }

    @DSLAction(displayed = true)
    String describe() {
        Stream<String> propertyKeys = model.propertyKeys().stream().filter(it -> !model.isStandardIdKey(it))
                                            .map(it -> new PropertyKeyImpl(model, it.name()).describeCoreDefinition());
        Stream<String> metaProperties = model.propertyKeys().stream().filter(it -> !model.isStandardIdKey(it))
                .map(it -> new PropertyKeyImpl(model, it.name()).describeMetaProperties()).filter(p->p != null);
        Stream<String> edgeLabels = model.edgeLabels().stream().map(it -> new EdgeLabelImpl(model,it.name()).describeEdgeLabel());
        Stream<String> vertexLabels = model.vertexLabels().stream().map(it -> vertexLabel(it.name()).describe());
        Stream<String> connections = model.edgeLabels().stream().map(it -> new EdgeLabelImpl(model,it.name()).describeConnections());

        return concat(concat(concat(concat(propertyKeys, metaProperties), edgeLabels),vertexLabels), connections).collect(joining(NL));

    }

    String clear()
    {
        drop();
        return "'clear()' is deprecated and will be removed in a future release. Please use 'drop()' instead.";
    }

    void drop()
    {
        model.drop();
    }

    GraphTraversalSource traversal() {
        return model.traversal();
    }

    Configuration config() {
        return new Configuration(model.configuration());
    }

    String getEffectiveSchemaMode()
    {
        return model.getEffectiveSchemaMode();
    }

    boolean getEffectiveAllowScan()
    {
        return model.getEffectiveAllowScan();
    }

    public static String toCommaSeparatedList(Stream<? extends Named> schemaElements) {
        return schemaElements.map(element -> quote(element)).collect(joining(", "));
    }

    public static String quote(Named schemaElement) {
        return quote(schemaElement.name());
    }

    public static String quote(String name) {
        return QM + name + QM;
    }

}
