/**
 * Copyright DataStax, Inc.
 *
 * Please see the included license file for details.
 */
package com.datastax.bdp.graph.api.schema;

import java.time.Duration;
import java.util.*;

import com.datastax.bdp.graph.api.model.*;
import com.datastax.bdp.graph.api.model.EdgeLabel;
import com.datastax.bdp.graph.api.model.PropertyKey;
import com.datastax.bdp.graph.api.model.Schema;
import com.datastax.bdp.graph.api.model.VertexLabel;
import com.github.misberner.duzzt.annotations.DSLAction;
import com.github.misberner.duzzt.annotations.GenerateEmbeddedDSL;
import com.github.misberner.duzzt.annotations.SubExpr;
import com.google.common.base.Preconditions;

import static com.datastax.bdp.graph.api.schema.SchemaImpl.*;
import static java.util.stream.Collectors.joining;

@GenerateEmbeddedDSL(
        name = "VertexLabel",
        syntax = "((partitionKey clusteringKey?)? properties? ttl? ifNotExists? create) | <index> | (properties (add|drop)) | <cache> | describe | exists | drop",
        where = {
                @SubExpr(name = "index", definedAs="index (" +
                        "(materialized by <add>)|" +
                        "(secondary by <add>)|" +
                        "(search ((properties (drop|remove))|((by (asText|asString|asStringAndText|withError)?)+ <add>)))|" +
                        "(property by <add>)|" +
                        "((outE|inE|bothE) by <add>)|" +
                        "(remove|drop)" +
                        ")"),
                @SubExpr(name = "add", definedAs="ifNotExists? add"),
                @SubExpr(name = "cache", definedAs="cache (properties|bothE) (ttl add | (remove|drop))"),
        })
public class VertexLabelImpl
{
    private enum ElementType {
        PROPERTY, EDGE;
    }

    private final String vertexLabelName;
    private final Schema schema;

    private VertexIndex.Builder vertexIndexBuilder;
    private VertexLabel.Builder vertexLabelBuilder;
    private String indexName;
    private EdgeIndex.Builder edgeIndexBuilder;
    private List<String> propertyKeys = new ArrayList<>();
    private List<String> edgeLabels = new ArrayList<>();
    private PropertyIndex.Builder propertyIndexBuilder;
    private boolean cache = false;
    private ElementType cacheType = null;
    private Duration ttl = null;
    private String propertyKey;


    VertexLabelImpl(Schema schema, String name) {
        this.schema = schema;
        vertexLabelName = name;
        vertexLabelBuilder = schema.buildVertexLabel(vertexLabelName);
    }

    @DSLAction(displayed = true)
    boolean exists()
    {
        return schema.vertexLabel(vertexLabelName) != null;
    }


    @DSLAction(displayed = true)
    void index(String name) {
        this.indexName = name;
    }

    @DSLAction(displayed = true)
    void materialized() {
        vertexIndexBuilder = getVertexLabel().buildVertexIndex(indexName).materialized();
    }

    @DSLAction(displayed = true)
    void secondary() {
        vertexIndexBuilder = getVertexLabel().buildVertexIndex(indexName).secondary();
    }

    @DSLAction(displayed = true)
    void search() {
        vertexIndexBuilder = getVertexLabel().buildVertexIndex(indexName).search();
    }

    @DSLAction(displayed = true)
    void by(String propertyKey) {
        this.propertyKey = propertyKey;
        if(vertexIndexBuilder != null) {
            vertexIndexBuilder = vertexIndexBuilder.byPropertyKey(propertyKey);
        } else if(propertyIndexBuilder != null) {
            propertyIndexBuilder = propertyIndexBuilder.byMetaPropertyKey(propertyKey);
        } else if(edgeIndexBuilder != null) {
            edgeIndexBuilder = edgeIndexBuilder.byPropertyKey(propertyKey);
        } else throw new AssertionError("Unplanned invocation");
    }

    @DSLAction(displayed = true)
    void asText()
    {
        vertexIndexBuilder = vertexIndexBuilder.byPropertyKeyAsText(propertyKey, VertexIndex.IndexOption.Text.FULLTEXT);
    }

    @DSLAction(displayed = true)
    void asString()
    {
        vertexIndexBuilder = vertexIndexBuilder.byPropertyKeyAsText(propertyKey, VertexIndex.IndexOption.Text.STRING);
    }

    @DSLAction(displayed = true)
    void asStringAndText()
    {
        vertexIndexBuilder = vertexIndexBuilder.byPropertyKeyAsStringAndText(propertyKey);
    }

    @DSLAction(displayed = true)
    void withError(double maxDistErr, double distErrPct)
    {
        vertexIndexBuilder = vertexIndexBuilder.byPropertyKeyWithError(propertyKey, maxDistErr, distErrPct);
    }

    @DSLAction(displayed = true)
    void outE(String label)
    {
        if (indexName != null)
        {
            edgeIndexBuilder = getVertexLabel().buildEdgeIndex(indexName, label).out();
        }
    }

    @DSLAction(displayed = true)
    void inE(String label)
    {
        if (indexName != null)
        {
            edgeIndexBuilder = getVertexLabel().buildEdgeIndex(indexName, label).in();
        }
    }

    @DSLAction(displayed = true)
    void cache() {
        this.cache = true;
    }

    @DSLAction(displayed = true)
    void bothE(String label) {
        if (cache) {
            cacheType = ElementType.EDGE;
            edgeLabels.add(label);
        } else {
            edgeIndexBuilder = getVertexLabel().buildEdgeIndex(indexName, label).both();
        }
    }

    @DSLAction(displayed = true)
    void bothE(String... labels) {
        if (labels!=null && labels.length==1) {
            bothE(labels[0]);
        } else if (labels!=null && labels.length>0) {
            edgeLabels.addAll(Arrays.asList(labels));
        }
        if (cache) cacheType = ElementType.EDGE;
        else throw new IllegalArgumentException("Need to specify exactly one edge label when building an edge index");
    }

    @DSLAction(displayed = true)
    void properties(String first, String... properties) {
        propertyKeys.add(first);
        if (properties!=null && properties.length>0) {
            propertyKeys.addAll(Arrays.asList(properties));
        }
        if (cache) throw new IllegalArgumentException("Can only cache ALL properties - not a subset");
    }

    @DSLAction(displayed = true)
    void properties() {
        if (cache) cacheType = ElementType.PROPERTY;
    }


    @DSLAction(displayed = true)
    void property(String propertyKey) {
        propertyIndexBuilder = getVertexLabel().buildPropertyIndex(indexName, propertyKey);
    }

    @DSLAction(displayed = true)
    void ttl(int timeToLiveInSeconds) {
        Preconditions.checkArgument(timeToLiveInSeconds>0,"Time-to-live is specified in seconds and must be a positive number. Given: %s",timeToLiveInSeconds);
        ttl = Duration.ofSeconds(timeToLiveInSeconds);
        vertexLabelBuilder = vertexLabelBuilder.ttl(ttl);
    }

    @DSLAction(displayed = true)
    void ifNotExists() {
        vertexLabelBuilder = vertexLabelBuilder.ifNotExists();

        if(vertexIndexBuilder != null) {
            vertexIndexBuilder = vertexIndexBuilder.ifNotExists();
        }
        if(propertyIndexBuilder != null) {
            propertyIndexBuilder = propertyIndexBuilder.ifNotExists();
        }
        if(edgeIndexBuilder != null) {
            edgeIndexBuilder = edgeIndexBuilder.ifNotExists();
        }
    }

    @DSLAction(displayed = true)
    void partitionKey(String first, String... properties)
    {
        vertexLabelBuilder = vertexLabelBuilder.partitionId(first);
        for (String pk : properties)
        {
            vertexLabelBuilder = vertexLabelBuilder.partitionId(pk);
        }
    }

    @DSLAction(displayed = true)
    void clusteringKey(String first, String... properties)
    {
        vertexLabelBuilder = vertexLabelBuilder.clusteringId(first);
        for (String pk : properties)
        {
            vertexLabelBuilder = vertexLabelBuilder.clusteringId(pk);
        }
    }

//    @DSLAction(terminator = true)
//    List<String> properties() {
//        return schema.vertexLabel(vertexLabelName).propertyKeys().stream().map(it->it.name()).collect(Collectors.toList());
//    }
//
//    @DSLAction(terminator = true)
//    Optional<Duration> ttl() {
//        return schema.vertexLabel(vertexLabelName).ttl();
//    }

    @DSLAction(displayed = true)
    void create() {
        propertyKeys.forEach( pk -> vertexLabelBuilder.addPropertyKeys(pk));
        vertexLabelBuilder.add();
    }

    @DSLAction(displayed = true)
    void add() {
        propertyKeys.forEach(it -> getVertexLabel().addPropertyKey(it));
        if(vertexIndexBuilder != null) {
            vertexIndexBuilder.add();
        }
        if(propertyIndexBuilder != null) {
            propertyIndexBuilder.add();
        }
        if(edgeIndexBuilder != null) {
            edgeIndexBuilder.add();
        }
        if (cacheType != null) {
            if (cacheType == ElementType.EDGE) {
                getVertexLabel().setEdgeCacheTime(ttl,edgeLabels.toArray(new String[edgeLabels.size()]));
            } else if (cacheType == ElementType.PROPERTY) {
                getVertexLabel().setPropertyCacheTime(ttl);
            } else throw new AssertionError("Invalid cache type: " + cacheType);
        }
    }

    @DSLAction(displayed = true)
    String remove()
    {
        drop();
        if (null != indexName || null != cacheType)
        {
            // we only need to issue a deprecation msg for the index.remove() or cache.remove() usage
            return "'remove()' is deprecated and will be removed in a future release. Please use 'drop()' instead.";
        }
        return "";
    }

    @DSLAction(displayed = true)
    void drop()
    {
        if (indexName != null)
        {
            VertexIndex vertexIndex = getVertexLabel().vertexIndex(indexName);
            if (vertexIndex != null)
            {
                if (propertyKeys.isEmpty())
                {
                    vertexIndex.drop();
                }
                else
                {
                    propertyKeys.forEach(it -> vertexIndex.dropPropertyKey(it));
                }
            }
            EdgeIndex edgeIndex = getVertexLabel().edgeIndex(indexName);
            if (edgeIndex != null)
            {
                edgeIndex.drop();
            }
            PropertyIndex propertyIndex = getVertexLabel().propertyIndex(indexName);
            if (propertyIndex != null)
            {
                propertyIndex.drop();
            }
        }
        else if (cacheType != null)
        {
            if (cacheType == ElementType.EDGE)
            {
                getVertexLabel().setEdgeCacheTime(Duration.ZERO, edgeLabels.toArray(new String[edgeLabels.size()]));
            }
            else if (cacheType == ElementType.PROPERTY)
            {
                if (!propertyKeys.isEmpty())
                    throw new IllegalArgumentException("Can only cache ALL properties - not a subset");
                getVertexLabel().setPropertyCacheTime(Duration.ZERO);
            }
            else throw new AssertionError("Invalid cache type: " + cacheType);
        }
        else if (!propertyKeys.isEmpty())
        {
            for (String key : propertyKeys)
            {
                getVertexLabel().dropPropertyKey(key);
            }
        }
        else
        {
            getVertexLabel().drop();
        }
    }

    @DSLAction(displayed = true)
    String describe() {

        VertexLabel vl = getVertexLabel();
        StringBuilder s = new StringBuilder();
        s.append(vertexLabelDescription(vl));
        Set<? extends IdPropertyKey> idKeys = vl.idPropertyKeys();
        if (!idKeys.isEmpty() && !vl.hasStandardId()) {
            s.append(".partitionKey(");
            s.append(toCommaSeparatedList(idKeys.stream().filter(it -> it.getType() == IdPropertyKey.Type.Partition)));
            s.append(")");
            if (idKeys.stream().filter( it -> it.getType() == IdPropertyKey.Type.Clustering).count() > 0) {
                s.append(".clusteringKey(");
                s.append(toCommaSeparatedList(idKeys.stream().filter(it -> it.getType() == IdPropertyKey.Type.Clustering)));
                s.append(")");
            }
        }
        Set<PropertyKey> remainingProperties = new LinkedHashSet<>(vl.propertyKeys());
        remainingProperties.removeAll(idKeys);
        if (!remainingProperties.isEmpty()) {
            s.append(".properties(");
            s.append(toCommaSeparatedList(remainingProperties.stream()));
            s.append(")");
        }
        if (vl.ttl().isPresent()) {
            s.append(".ttl(").append(vl.ttl().get().getSeconds()).append(")");
        }
        s.append(".create()");
        vl.vertexIndices().forEach(it-> {
            s.append("\n" + vertexLabelDescription(vl));
            s.append(".index(" + quote(it) + ")." + it.getType().name().toLowerCase() + "()");
            s.append(it.propertyKeys().stream().map(p -> ".by(" + quote(p) + ")" + searchParamsDescription(it, p)).collect(joining()));
            s.append(".add()");
        });

        vl.edgeIndices().forEach(it-> {
            s.append("\n" + vertexLabelDescription(vl));
            s.append(".index(" + quote(it) + ")." + it.direction().name().toLowerCase() + "E(" + quote(it.edgeLabel()) + ")");
            s.append(it.propertyKeys().stream().map(p->".by(" + quote(p) + ")").collect(joining()));
            s.append(".add()");
        });

        vl.propertyIndices().forEach(it-> {
            s.append("\n" + vertexLabelDescription(vl));
            s.append(".index(" + quote(it) + ").property(" + quote(it.propertyKey()) + ")");
            s.append(it.propertyKeys().stream().map(p->".by(" + quote(p) + ")").collect(joining()));
            s.append(".add()");
        });

        vl.cacheConfigs().forEach( it -> {
            s.append("\n" + vertexLabelDescription(vl));
            s.append(".cache()");
            if (it.isPropertyCache()) {
                s.append(".properties()");
            } else {
                EdgeLabel el = it.edgeLabel();
                s.append(".bothE(");
                if (el != null) s.append(quote(el));
                s.append(")");
            }
            s.append(".ttl(").append(it.cacheTime().getSeconds()).append(").add()");
        });

        return s.toString();
    }

    private String searchParamsDescription(VertexIndex it, IndexedPropertyKey p)
    {
        if(p.getIndexOption().isDescriptionSuppressed()) {
            return "";
        }
        return p.getIndexOption().describe();
    }

    private String vertexLabelDescription(VertexLabel vertexLabel) {return SchemaImpl.VERTEX_LABEL_PREFIX + quote(vertexLabel) + ")";}


    private VertexLabel getVertexLabel() {
        VertexLabel vertexLabel = schema.vertexLabel(vertexLabelName);
        if (vertexLabel == null) throw new IllegalArgumentException("Vertex label does not exist: " + vertexLabelName);
        return vertexLabel;
    }

}
